from typing import cast

from sendr_utils import alist

from mail.payments.payments.core.actions.arbitrage.notify import NotifyArbitrageAction
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get_acquirer import GetAcquirerMerchantAction
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.actions.tlog.refund import ExportRefundToTLogAction
from mail.payments.payments.core.entities.arbitrage import Arbitrage
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.enums import RefundStatus
from mail.payments.payments.core.entities.log import RefundStatusUpdatedLog
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.utils.datetime import utcnow


class UpdateRefundAction(APICallbackTaskMixin, BaseDBAction):
    transact = True

    def __init__(self, refund: Order):
        assert refund.shop is not None
        super().__init__()
        self.refund: Order = refund

    async def handle(self) -> Order:
        refund = self.refund

        assert (
            refund.trust_refund_id is not None
            and refund.order_id is not None
            and refund.original_order_id is not None
        )

        # Refund must be requested first
        if refund.refund_status != RefundStatus.REQUESTED:
            return refund

        acquirer = await GetAcquirerMerchantAction(uid=refund.uid).run()
        acquirer = refund.get_acquirer(acquirer)

        assert refund.shop
        trust = self.clients.get_trust_client(refund.uid, refund.shop.shop_type)
        # Getting current status from trust
        trust_refund_status = (
            await trust.refund_get(uid=refund.uid, acquirer=acquirer, refund_id=refund.trust_refund_id)
        )['status']
        refund_status = RefundStatus.from_trust(trust_refund_status)

        # Refund status didn't change, setting `updated` to now. No need to log this change
        if refund.refund_status == refund_status:
            return await self.storage.order.save(refund)

        merchant = await self.storage.merchant.get(refund.uid)

        try:
            arbitrage = await self.storage.arbitrage.get_by_refund_id(uid=refund.uid, refund_id=refund.order_id)
            await NotifyArbitrageAction(arbitrage_id=cast(int, arbitrage.arbitrage_id)).run_async()
        except Arbitrage.DoesNotExist:
            pass

        await SendToHistoryOrderAction(uid=refund.uid, order_id=refund.original_order_id).run_async()

        # Updating refund
        refund.refund_status = refund_status
        if refund_status == RefundStatus.COMPLETED:
            await ExportRefundToTLogAction(uid=refund.uid, order_id=refund.order_id).run_async()
            refund.closed = utcnow()
        refund = await self.storage.order.save(refund)
        assert refund.original_order_id is not None

        # Scheduling callbacks
        await self.create_refund_status_task(refund, merchant=merchant)
        if refund.service_client_id is not None and refund.service_merchant_id is not None:
            service = await self.storage.service.get_by_related(
                service_client_id=refund.service_client_id,
                service_merchant_id=refund.service_merchant_id,
            )
            await self.create_refund_status_task(refund, service=service)

        # Logging to change_log and YT
        assert (
            refund.revision is not None
            and refund.refund_status is not None
            and refund.order_id is not None
            and refund.original_order_id is not None
        )

        await self.storage.change_log.create(ChangeLog(
            uid=refund.uid,
            revision=refund.revision,
            operation=OperationKind.UPDATE_REFUND,
            arguments={'refund_status': refund.refund_status.value},
        ))

        refund.items = await alist(self.storage.item.get_for_order(uid=refund.uid, order_id=refund.order_id))
        await self.pushers.log.push(RefundStatusUpdatedLog(
            merchant_name=merchant.name,
            merchant_uid=refund.uid,
            merchant_acquirer=acquirer,
            refund_id=refund.order_id,
            order_id=refund.original_order_id,
            status=refund.refund_status.value,
            price=refund.log_price
        ))

        return refund
