from datetime import datetime
from typing import Any, Dict, Optional, Tuple

from sendr_interactions import exceptions as interaction_errors
from sendr_utils import alist, enum_value, frommsktimestamp

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.interactions.trust import (
    GetPaymentInfoInTrustAction, UnholdPaymentInTrustAction
)
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.mixins.moderation import MerchantModerationMixin
from mail.payments.payments.core.actions.moderation import ScheduleOrderModerationAction
from mail.payments.payments.core.actions.order.clear import ClearOrderAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.actions.tlog.order import ExportOrderToTLogAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.enums import FunctionalityType, OperationKind, OrderSource, PayStatus
from mail.payments.payments.core.entities.log import OrderStatusUpdatedLog
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.moderation import Moderation
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service, ServiceMerchant
from mail.payments.payments.core.entities.transaction import Transaction, TransactionStatus
from mail.payments.payments.core.exceptions import (
    CoreActionDenyError, CoreFailError, CoreInteractionError, OrdersAmountExceed
)
from mail.payments.payments.storage.exceptions import ModerationNotFound
from mail.payments.payments.utils.datetime import utcnow


class UpdateTransactionAction(APICallbackTaskMixin, MerchantModerationMixin, BaseDBAction):
    """
    Обновляем состояние транзакции и заказа.
    Идентифицируем текущее состояние заказа и транзакции по order.pay_status и transaction.status.
    В зависимости от состояния задействуется либо логика слежения за статусом платежа,
    либо логика ожидания результата модерации.
    """

    transact = True

    def __init__(self,
                 transaction: Transaction,
                 merchant: Optional[Merchant] = None,
                 ):
        super().__init__()
        self.transaction: Transaction = transaction
        self.merchant: Optional[Merchant] = merchant
        self.order: Optional[Order] = transaction.order if isinstance(transaction.order, Order) else None

    async def _log_status_update(self,
                                 order: Order,
                                 transaction: Transaction,
                                 merchant: Merchant,
                                 service: Optional[Service] = None,
                                 ) -> None:
        assert order.order_id is not None \
               and order.pay_status is not None \
               and transaction.trust_purchase_token is not None

        service_id = service.service_id if service else None
        service_name = service.name if service else None
        order.items = await alist(self.storage.item.get_for_order(uid=order.uid, order_id=order.order_id))
        log = OrderStatusUpdatedLog(
            merchant_name=merchant.name,
            merchant_uid=merchant.uid,
            merchant_acquirer=order.get_acquirer(merchant.acquirer),
            order_id=order.order_id,
            purchase_token=transaction.trust_purchase_token,
            status=order.pay_status.value,
            customer_uid=order.customer_uid,
            service_id=service_id,
            service_name=service_name,
            sdk_api_created=order.created_by_source == OrderSource.SDK_API,
            sdk_api_pay=order.pay_by_source == OrderSource.SDK_API,
            created_by_source=order.created_by_source,
            pay_by_source=order.pay_by_source,
            merchant_oauth_mode=enum_value(order.merchant_oauth_mode),
            price=order.log_price
        )
        await self.pushers.log.push(log)

    async def _log_order_updated_into_changelog(self, order: Order) -> None:
        assert order.revision is not None and order.pay_status is not None
        change_log = ChangeLog(uid=order.uid,
                               revision=order.revision,
                               operation=OperationKind.UPDATE_ORDER,
                               arguments={
                                   'order_id': order.order_id,
                                   'merchant_uid': order.uid,
                                   'order_pay_status': order.pay_status.value,
                               })
        await self.storage.change_log.create(change_log)

    async def _merchant_approved(self, merchant: Merchant) -> bool:
        """Обернуть try-except, чтобы вернуть флаг проверки."""
        try:
            await self.require_moderation(merchant=merchant, functionality_type=FunctionalityType.PAYMENTS)
            return True
        except CoreActionDenyError:
            return False

    async def _order_moderation_enabled(self, order: Order) -> bool:
        if order.is_subscription:
            return False

        if order.service_merchant_id is None:
            return True

        service_merchant = await self.storage.service_merchant.get(service_merchant_id=order.service_merchant_id)
        return service_merchant.service.order_moderation_enabled  # type: ignore

    @staticmethod
    def _merchant_moderation_enabled(merchant: Merchant) -> bool:
        moderation_disabled = settings.ORDER_MODERATION_DISABLED
        if moderation_disabled or merchant.trustworthy:
            return False
        return True

    async def _need_moderation_status_before_clear(self, merchant: Merchant, order: Order) -> bool:
        """Нужно знать статус модерации заказа перед CLEAR? В будущем нужно уметь это делать на уровне заказа."""
        need_merchant_moderation = self._merchant_moderation_enabled(merchant)
        need_order_moderation = await self._order_moderation_enabled(order)
        return need_order_moderation and need_merchant_moderation

    def _on_transaction_failed(self, transaction: Transaction, order: Order, trust_payment: Dict) -> None:
        order.pay_status = PayStatus.REJECTED
        transaction.trust_resp_code = trust_payment.get('payment_resp_code')
        transaction.trust_failed_result = ' :: '.join([
            trust_payment.get('payment_resp_code', ''),
            trust_payment.get('payment_resp_desc', ''),
        ])

    async def _on_transaction_cleared(self, order: Order, transaction: Transaction, paymethod_id: str,
                                      closed: datetime) -> None:
        assert transaction.tx_id

        order.pay_status = PayStatus.PAID
        order.paymethod_id = paymethod_id
        order.closed = closed

        await ExportOrderToTLogAction(uid=order.uid, tx_id=transaction.tx_id).run_async()

    def _on_transaction_cancelled(self, order: Order) -> None:
        """Считаем, что если pay_status = moderation_negative, то cancel произошел по причине модерации."""
        if order.pay_status == PayStatus.IN_CANCEL:
            order.pay_status = PayStatus.CANCELLED
            order.closed = utcnow()
        elif order.pay_status != PayStatus.MODERATION_NEGATIVE:
            order.pay_status = PayStatus.REJECTED

    async def _request_clear(self, transaction: Transaction, order: Order) -> None:
        order.pay_status = PayStatus.IN_PROGRESS
        self.logger.context_push(new_order_pay_status=order.pay_status.value)
        self.logger.info('Requesting clear')

        try:
            await ClearOrderAction(transaction=transaction, order=order).run()
        except OrdersAmountExceed:
            self.logger.info('Orders amount was exceeded')
            await self._request_unhold(pay_status=PayStatus.IN_CANCEL, transaction=transaction, order=order)

    async def _request_unhold(self, pay_status: PayStatus, transaction: Transaction, order: Order) -> None:
        order.pay_status = pay_status

        self.logger.context_push(new_order_pay_status=order.pay_status)
        self.logger.info('Requesting unhold')
        assert (
            transaction.trust_purchase_token is not None
            and self.merchant is not None
            and self.merchant.acquirer is not None
        )
        await UnholdPaymentInTrustAction(
            order=order,
            purchase_token=transaction.trust_purchase_token,
            acquirer=order.get_acquirer(self.merchant.acquirer),  # type: ignore
        ).run()

    async def _on_order_held(self, transaction: Transaction, order: Order, service: Optional[Service]) -> None:
        assert not service or isinstance(service.service_merchant, ServiceMerchant)
        if order.service_merchant_id and (not service or not service.service_merchant.enabled):  # type: ignore
            self.logger.info(f'Unhold because {"service absent" if not service else "service_merchant disabled"}')
            await self._request_unhold(pay_status=PayStatus.IN_CANCEL, transaction=transaction, order=order)
        elif order.autoclear:
            # Skipping HELD straight to IN_PROGRESS
            await self._request_clear(transaction=transaction, order=order)
        else:
            order.pay_status = PayStatus.HELD
            transaction.poll = False
            self.logger.context_push(new_order_pay_status=order.pay_status.value)
            self.logger.info('Autoclear disabled. Awaiting manual clear/cancel')

    async def _on_transaction_held(self,
                                   transaction: Transaction,
                                   order: Order,
                                   merchant: Merchant,
                                   service: Optional[Service]) -> None:
        """Обработка активной транзакции по событию HELD в трасте."""
        order.held_at = utcnow()
        assert order.order_id
        need_moderation = await self._need_moderation_status_before_clear(merchant=merchant, order=order)
        if not need_moderation:
            self.logger.info('Moderation is not required')
            return await self._on_order_held(transaction=transaction, order=order, service=service)
        self.logger.info('Moderation is required')

        try:
            moderation = await self.storage.moderation.get_for_order(uid=order.uid, order_id=order.order_id)
            self.logger.context_push(moderation_id=moderation.moderation_id, approved=moderation.approved)
            if moderation.approved is None:
                # Must not happen since order.pay_status is not IN_MODERATION
                self.logger.error('Bad order moderation state: '
                                  'active transaction for order with unresolved moderation')
                raise CoreFailError('Bad moderation state')
            elif moderation.approved:
                self.logger.info('Moderation passed')
                return await self._on_order_held(transaction=transaction, order=order, service=service)
            else:
                self.logger.info('Moderation failed')
                await self._request_unhold(pay_status=PayStatus.MODERATION_NEGATIVE,
                                           transaction=transaction,
                                           order=order)
        except ModerationNotFound:
            self.logger.info('Moderation is not found')
            await ScheduleOrderModerationAction(order=order).run()
            order.pay_status = PayStatus.IN_MODERATION

    async def _sync_transaction_status(self,
                                       transaction: Transaction,
                                       order: Order,
                                       merchant: Merchant,
                                       trust_payment: Dict,
                                       service: Optional[Service]) -> bool:
        """Обновление статуса транзакции на основании статуса платежа из траста.
        Returns:
            флаг изменения состояния - если перехода в новый статус нет - False, иначе True
        """
        assert order.pay_status
        try:
            status = TransactionStatus.from_trust(trust_payment['payment_status'])
        except KeyError:  # some unknown status
            self.logger.exception('Unable get transaction status from trust status')
            return False

        self.logger.context_push(trust_status=enum_value(status))
        if not transaction.status.allowed_next_status(status):
            self.logger.info('Transaction next status is not allowed')
            return False

        self.logger.info('Got next transaction status')
        if status == TransactionStatus.HELD:
            await self._on_transaction_held(transaction=transaction, order=order, merchant=merchant, service=service)
        elif status == TransactionStatus.CLEARED:
            await self._on_transaction_cleared(
                order=order,
                transaction=transaction,
                paymethod_id=trust_payment.get('paymethod_id', order.paymethod_id),
                closed=frommsktimestamp(float(trust_payment['clear_real_ts']))
            )
        elif status == TransactionStatus.FAILED:
            self._on_transaction_failed(transaction=transaction, order=order, trust_payment=trust_payment)
        elif status == TransactionStatus.CANCELLED:
            self._on_transaction_cancelled(order=order)

        transaction.status = status
        transaction.trust_terminal_id = trust_payment.get('terminal', {}).get('id')

        self.logger.context_push(
            new_order_pay_status=order.pay_status.value,
            new_transaction_status=status.value,
        )
        self.logger.info('Transaction status synced')
        return True

    async def _on_moderation_result(self,
                                    transaction: Transaction,
                                    order: Order,
                                    moderation: Moderation,
                                    service: Optional[Service]) -> None:
        """clear/unhold по результату модерации."""
        merchant = await self.storage.merchant.get(uid=order.uid)
        if await self._merchant_approved(merchant=merchant) and moderation.approved:
            self.logger.info('Moderation passed')
            await self._on_order_held(transaction=transaction, order=order, service=service)
        else:
            self.logger.info('Moderation failed')
            await self._request_unhold(PayStatus.MODERATION_NEGATIVE, transaction=transaction, order=order)

    async def _when_order_in_moderation(self,
                                        transaction: Transaction,
                                        order: Order,
                                        service: Optional[Service]) -> bool:
        """Сверяемся со статусом модерации, чтобы обновить заказ и транзакцию."""
        # если мы здесь, то модерация должна существовать, но она может быть быть "pending"
        assert order.order_id

        moderation = await self.storage.moderation.get_for_order(uid=order.uid, order_id=order.order_id)
        if moderation.approved is None:
            self.logger.info('Order moderation is still in progress')
            return False
        else:
            self.logger.info('Order moderation is resolved')
            await self._on_moderation_result(transaction=transaction,
                                             order=order,
                                             moderation=moderation,
                                             service=service)
            return True

    async def _fetch_entites(self) -> Tuple[Merchant, Order, Transaction, Optional[Service]]:
        transaction = self.transaction
        uid = transaction.uid
        order_id = transaction.order_id

        order = self.order
        if order is None:
            order = await self.storage.order.get(uid, order_id, select_customer_subscription=None, for_update=True)

        merchant = self.merchant
        if merchant is None:
            self.merchant = merchant = await GetMerchantAction(uid=uid, skip_data=True, skip_oauth=True).run()

        service = None
        if order.service_client_id is not None and order.service_merchant_id is not None:
            service = await self.storage.service.get_by_related(
                service_client_id=order.service_client_id,
                service_merchant_id=order.service_merchant_id,
            )

        return merchant, order, transaction, service

    @staticmethod
    def _maybe_update_user_email(order: Order, trust_payment: Dict[str, Any]) -> bool:
        """
        Обновляем user_email: Trust источник правды по этому полю, т. к. email может быть передан не через нас
        PAYBACK-655
        """
        user_email: Optional[str] = trust_payment.get('user_email')
        if user_email is not None and user_email != order.user_email:
            order.user_email = user_email
            return True
        return False

    async def _update_transaction(self) -> Transaction:
        merchant, order, transaction, service = await self._fetch_entites()
        assert (
            transaction.trust_purchase_token is not None
            and transaction.status is not None
            and order.pay_status is not None
            and merchant.acquirer is not None
        )

        trust_payment = await GetPaymentInfoInTrustAction(
            purchase_token=transaction.trust_purchase_token,
            order=order,
            acquirer=order.get_acquirer(merchant.acquirer),  # type: ignore
        ).run()

        self.logger.context_push(
            uid=merchant.uid,
            order_id=order.order_id,
            tx_id=transaction.tx_id,
            service_merchant_id=order.service_merchant_id,
            order_pay_status=order.pay_status.value,
            transaction_status=transaction.status.value,
        )
        if service:
            self.logger.context_push(service_id=service.service_id)
        self.logger.info('Starting transaction update')

        if order.in_moderation:
            updated = await self._when_order_in_moderation(transaction=transaction, order=order, service=service)
        else:
            updated = await self._sync_transaction_status(
                transaction=transaction,
                order=order,
                merchant=merchant,
                trust_payment=trust_payment,
                service=service,
            )

        user_email_updated: bool = self._maybe_update_user_email(order, trust_payment)

        if not updated:
            if user_email_updated:
                await self.storage.order.save(order)
            transaction.increment_check_tries()
            transaction = await self.storage.transaction.save(transaction)
            self.logger.info('Transaction is updated')
            return transaction

        order = await self.storage.order.save(order)
        order = await self.storage.order.get(
            order.uid,
            order.order_id,
            select_customer_subscription=bool(order.customer_subscription_id)
        )
        assert order.order_id is not None
        await SendToHistoryOrderAction(uid=order.uid, order_id=order.order_id).run_async()
        transaction.reset_check_tries()
        transaction = await self.storage.transaction.save(transaction)

        if service:
            await self.create_order_status_task(order=order, service=service)
        await self.create_order_status_task(order=order, merchant=merchant)

        await self._log_status_update(order=order, transaction=transaction, merchant=merchant, service=service)
        await self._log_order_updated_into_changelog(order=order)
        self.logger.info('Order and transaction are updated')

        return transaction

    async def handle(self) -> Transaction:
        """Точка входа в логику обновления состояния транзакции.

        Синхронизации статуса транзакции со статусом платежа из траста
        (и принятие соответствующих действий) с одной стороны, либо обработка результа модерации заказа
        с другой стороны. Возможно стоит разделить логику на отдельные actions.

        """
        try:
            return await self._update_transaction()
        except (interaction_errors.BaseInteractionError, CoreInteractionError):
            transaction = self.transaction
            transaction.increment_check_tries()
            self.logger.exception('Unable to update transaction')
            return await self.storage.transaction.save(transaction)
