from typing import Optional

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.userinfo import UserInfo
from mail.payments.payments.core.exceptions import LoginNotFound, MerchantNotFoundError
from mail.payments.payments.interactions.exceptions import BlackBoxDefaultEmailNotFoundError, BlackBoxUserNotFoundError
from mail.payments.payments.storage.exceptions import MerchantNotFound


class BaseUserRoleAction(BaseDBAction):
    merchant: Optional[Merchant] = None

    def __init__(self, merchant_id: str, user_uid: Optional[int] = None, user_email: Optional[str] = None):
        super().__init__()
        self.merchant_id: str = merchant_id
        self.user_uid: Optional[int] = user_uid
        self.user_email: Optional[str] = user_email

    async def get_userinfo(self,
                           uid: Optional[int] = None,
                           login: Optional[str] = None,
                           get_default_emails: bool = True) -> UserInfo:
        try:
            userinfo = await self.clients.blackbox.userinfo(uids=uid, login=login,
                                                            get_default_emails=get_default_emails)
        except (BlackBoxUserNotFoundError, BlackBoxDefaultEmailNotFoundError):
            raise LoginNotFound(message='Unable to get user uid or user email to assign role')
        return userinfo

    async def pre_handle(self) -> None:
        await super().pre_handle()
        try:
            self.merchant = await self.storage.merchant.get_by_merchant_id(self.merchant_id)
        except MerchantNotFound:
            raise MerchantNotFoundError(merchant_id=self.merchant_id)
