from typing import ClassVar, Optional, Tuple

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.transact_email import TransactEmailAction
from mail.payments.payments.core.actions.user_role.base import BaseUserRoleAction
from mail.payments.payments.core.entities.enums import MerchantRole
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.task import Task
from mail.payments.payments.core.entities.user import User
from mail.payments.payments.core.entities.user_role import UserRole
from mail.payments.payments.core.exceptions import RoleAssignmentNotAllowed
from mail.payments.payments.storage.exceptions import UserNotFound, UserRoleNotFound


class CreateUserRoleAction(BaseUserRoleAction):
    """
    Создать роль пользователя в рамках мерчанта:
    ассоцияция один-к-одному между User и Merchant,
    аннотированная ролью пользователя, определяющей привелегии пользователя
    относительно возможных операций, совершаемых в рамках мерчанта.

    Если ассоциация уже существует, то будет перезаписан параметр роли пользователя.
    """
    transact = True
    required_merchant_roles: ClassVar[Optional[Tuple[MerchantRole, ...]]] = (MerchantRole.ADMIN,)

    def __init__(self,
                 merchant_id: str,
                 role: MerchantRole,
                 description: Optional[str] = None,
                 user_uid: Optional[int] = None,
                 user_email: Optional[str] = None,
                 notify: bool = False,
                 ):
        super().__init__(merchant_id=merchant_id, user_uid=user_uid, user_email=user_email)
        self.role: MerchantRole = role
        self.description: Optional[str] = description
        self.notify: bool = notify

    async def _notify_user_on_role(self, merchant: Merchant, user_email: str, role: MerchantRole) -> Task:
        """Notifies the user of the assigned role."""
        to_email = user_email
        mailing_id = settings.SENDER_MAILING_USER_ROLE_ASSIGNED
        render_context = {
            'user_role': role.value,
            'merchant_name': merchant.name
        }
        return await TransactEmailAction(
            to_email=to_email,
            render_context=render_context,
            mailing_id=mailing_id,
        ).run_async()

    def check_role_assignment_allowed(self, current_role: Optional[MerchantRole]) -> None:
        if (
            current_role != self.role
            and (current_role == MerchantRole.OWNER or self.role == MerchantRole.OWNER)
        ):
            raise RoleAssignmentNotAllowed

    async def handle(self) -> UserRole:
        assert (self.user_uid is None) != (self.user_email is None), \
            'Exactly one of "user_uid", "user_email" is required'

        if self.user_email is not None:
            userinfo = await self.get_userinfo(login=self.user_email)
        else:
            userinfo = await self.get_userinfo(uid=self.user_uid)
            self.user_email = userinfo.default_email
        self.user_uid, default_user_email = userinfo.uid, userinfo.default_email
        assert self.user_email is not None and default_user_email is not None

        # Getting or creating user
        try:
            user = await self.storage.user.get(uid=self.user_uid)
        except UserNotFound:
            user = User(uid=self.user_uid, email=default_user_email)
            user = await self.storage.user.create(user)

        # Getting or creating user_role
        try:
            user_role = await self.storage.user_role.get(uid=self.user_uid,
                                                         merchant_id=self.merchant_id,
                                                         for_update=True)
            self.check_role_assignment_allowed(current_role=user_role.role)
            user_role.role = self.role
            user_role.description = self.description
            user_role = await self.storage.user_role.save(user_role)
        except UserRoleNotFound:
            self.check_role_assignment_allowed(current_role=None)
            user_role = UserRole(
                uid=user.uid,
                merchant_id=self.merchant_id,
                role=self.role,
                description=self.description,
                email=self.user_email,
            )
            user_role = await self.storage.user_role.create(user_role)

        if self.notify:
            assert self.merchant
            await self._notify_user_on_role(self.merchant, default_user_email, self.role)
        return user_role


class CreateOwnerUserRoleAction(CreateUserRoleAction):
    # используем внутри регистрации, а не как основной функционал АПИ-ручки - не нужна проверка ролей
    required_merchant_roles: ClassVar[Optional[Tuple[MerchantRole, ...]]] = None

    def __init__(self,
                 merchant_id: str,
                 user_uid: Optional[int] = None,
                 user_email: Optional[str] = None,
                 notify: bool = False,
                 ):
        super().__init__(
            merchant_id=merchant_id,
            role=MerchantRole.OWNER,
            user_uid=user_uid,
            user_email=user_email,
            notify=notify
        )

    def check_role_assignment_allowed(self, current_role: Optional[MerchantRole]) -> None:
        if current_role not in (None, MerchantRole.OWNER) or self.role != MerchantRole.OWNER:
            raise RoleAssignmentNotAllowed
