from typing import List, Optional

from sendr_utils import alist

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.user_role.base import BaseUserRoleAction
from mail.payments.payments.core.entities.enums import MerchantRole
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.user_role import UserRole


class GetUserRolesForMerchantAction(BaseUserRoleAction):
    """
    Показать все UserRoles, ассоциированные с одним Merchant.
    """

    required_merchant_roles = (MerchantRole.VIEWER,)

    def __init__(self, merchant_id: str, role: Optional[MerchantRole] = None):
        super().__init__(merchant_id=merchant_id)
        self.role: Optional[MerchantRole] = role

    async def handle(self) -> List[UserRole]:
        return await alist(self.storage.user_role.find(
            merchant_id=self.merchant_id,
            role=self.role,
        ))


class GetUserRolesForUserAction(BaseDBAction):
    """
    Показать все UserRoles, ассоциированные с одним User.
    """
    def __init__(self, uid: int, role: Optional[MerchantRole] = None):
        super().__init__()
        self.uid: int = uid
        self.role: Optional[MerchantRole] = role

    async def handle(self) -> List[UserRole]:
        user_roles = []
        async for role in self.storage.user_role.find(uid=self.uid, role=self.role, with_merchant=True):
            assert isinstance(role.merchant, Merchant)
            role.merchant = await GetMerchantAction(merchant=role.merchant).run()
            user_roles.append(role)
        return user_roles
