from typing import Any, Dict, Optional

from sendr_utils import enum_value

from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.core.entities.enums import APICallbackSignMethod, CallbackMessageType
from mail.payments.payments.core.entities.merchant import APICallbackParams
from mail.payments.payments.interactions.service import ServiceClient


class CallbackWorkerAction(BaseAction):
    """
    Обработчик задач на отправку callback-сообщений.
    Ожидает увидеть в параметрах задачи некоторое сообщение, которое нужно отправить по заданному адресу.
    В зависимости от наличия TVM ID запрос будет отправлен через Zora-клиент (нет TVM ID),
    либо через TVM Client, если указан.
    """

    def __init__(self,
                 tvm_id: Optional[int] = None,
                 callback_url: Optional[str] = None,
                 callback_message_type: Optional[str] = None,
                 callback_params: Optional[APICallbackParams] = None,
                 message: Optional[Dict[str, Any]] = None,
                 ):
        super().__init__()
        self.tvm_id = tvm_id
        self.callback_message_type = CallbackMessageType(callback_message_type)
        self.callback_params = callback_params
        self.message = message
        self.callback_url = callback_url

    async def _handle_tvm_case(self) -> None:
        assert self.tvm_id is not None
        if self.callback_url:
            client = ServiceClient(tvm_id=self.tvm_id, logger=self.logger.clone(), request_id=None)
            try:
                json = {
                    "type": enum_value(self.callback_message_type),
                    "data": self.message
                }
                await client.callback_service(url=self.callback_url, json=json)
            finally:
                await client.close()

    async def _handle_no_tvm_case(self) -> None:
        if self.callback_url:
            assert self.message is not None
            if self.callback_params and self.callback_params.sign_method == APICallbackSignMethod.JWT:
                assert self.callback_params.secret is not None
                await self.clients.callback.post_jwt_message(
                    url=self.callback_url,
                    message=self.message,
                    secret=self.callback_params.secret
                )
            else:
                assert self.context.partner_crypto
                await self.clients.callback.post_signed_message(
                    url=self.callback_url,
                    message=self.message,
                    signer=self.context.partner_crypto.sign
                )

    async def handle(self) -> None:
        if self.tvm_id is not None:
            await self._handle_tvm_case()
        else:
            await self._handle_no_tvm_case()
