import time
from typing import Optional, Tuple, Type, Union, cast

from sendr_utils import enum_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.contract import InitContractAction
from mail.payments.payments.core.actions.init_products import InitProductsAction
from mail.payments.payments.core.actions.interactions.yandex_pay_admin import MerchantModerationResultNotifyAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.mixins.moderation import MerchantModerationMixin
from mail.payments.payments.core.actions.order.get import GetOrderAction
from mail.payments.payments.core.actions.transact_email import TransactEmailAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.enums import (
    PAYMETHOD_ID_OFFLINE, DocumentType, FunctionalityType, ModerationType, TaskType
)
from mail.payments.payments.core.entities.log import MerchantModerationApprovedLog
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.moderation import FastModerationRequest, Moderation
from mail.payments.payments.core.entities.not_fetched import NOT_FETCHED
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.entities.subscription import Subscription
from mail.payments.payments.core.entities.task import Task
from mail.payments.payments.core.exceptions import CoreFailError
from mail.payments.payments.storage.exceptions import ModerationNotFound, StorageNotFound, TransactionNotFound
from mail.payments.payments.storage.logbroker.consumers.moderation import ModerationResult
from mail.payments.payments.storage.logbroker.producers.moderation import (
    FastModerationRequestProducer, FastModerationResponseProducer, ModerationProducer
)
from mail.payments.payments.utils.datetime import utcnow
from mail.payments.payments.utils.helpers import decimal_format
from mail.payments.payments.utils.stats import (
    merchant_moderation_time, order_moderation_time, subscription_moderation_time
)


def choose_moderation_producer_cls(entity_fast_moderation: bool) -> Union[Type[ModerationProducer],
                                                                          Type[FastModerationRequestProducer]]:
    return FastModerationRequestProducer if settings.LB_ALLOW_FAST_MODERATION and entity_fast_moderation \
        else ModerationProducer


class StartMerchantModerationAction(MerchantModerationMixin, BaseDBAction):
    transact = True
    documents_for_moderation = {
        DocumentType.OFFER, DocumentType.OTHER, DocumentType.PASSPORT, DocumentType.PROXY, DocumentType.SIGNER_PASSPORT,
    }

    def __init__(self, merchant_uid: int, functionality_type: FunctionalityType = FunctionalityType.PAYMENTS):
        super().__init__()
        self.merchant_uid = merchant_uid
        if isinstance(functionality_type, str):
            functionality_type = FunctionalityType(functionality_type)
        self.functionality_type = functionality_type

    async def _get_or_create_moderation(self, merchant: Merchant) -> Moderation:
        moderation = await self.get_revision_moderation(merchant.uid, merchant.revision, self.functionality_type)
        if moderation:
            return moderation
        return await self.storage.moderation.create(Moderation(
            moderation_type=ModerationType.MERCHANT,
            functionality_type=self.functionality_type,
            uid=merchant.uid,
            revision=merchant.revision,
        ))

    async def _upload_documents(self, moderation: Moderation, merchant: Merchant) -> None:
        for document in merchant.documents:
            if document.document_type not in self.documents_for_moderation:
                continue
            _, data_iter = await self.clients.payments_mds.download(document.path)
            path = await self.clients.moderation_mds.upload(f'moderation_{moderation.moderation_id}', data_iter)
            document.moderation_url = self.clients.moderation_mds.download_url(path)

    async def handle(self) -> None:
        # Getting merchant and get-or-creating moderation instance
        merchant: Merchant = await GetMerchantAction(
            uid=self.merchant_uid, skip_moderation=True, skip_functionalities=True
        ).run()
        moderation = await self._get_or_create_moderation(merchant)

        # Uploading documents to a separate MDS namespace
        await self._upload_documents(moderation, merchant)

        # Change log
        await self.storage.change_log.create(ChangeLog(
            uid=merchant.uid,
            revision=merchant.revision,
            operation=OperationKind.START_MERCHANT_MODERATION,
            info={'moderation_id': moderation.moderation_id},
        ))

        # Writing moderation to LB
        producer_cls = choose_moderation_producer_cls(merchant.fast_moderation)
        async with producer_cls(self.context.lb_factory) as producer:
            await producer.write_merchant(moderation, merchant)


class StartOrderModerationWorkerAction(BaseDBAction):
    """Отправить модерацию заказа на обработку. Обработчик задач типа START_ORDER_MODERATION."""
    transact = True

    def __init__(self, moderation_id: int):
        super().__init__()
        self.moderation_id = moderation_id

    async def _log_moderation_scheduled(self, moderation: Moderation, order: Order) -> None:
        assert order.revision is not None
        change_log = ChangeLog(
            uid=order.uid,
            revision=order.revision,
            operation=OperationKind.START_ORDER_MODERATION,
            info={'moderation_id': moderation.moderation_id},
        )
        await self.storage.change_log.create(change_log)

    async def _fetch_entities(self, moderation_id: int) -> Tuple[Moderation, Order]:
        # Getting moderation
        moderation: Moderation = await self.storage.moderation.get(moderation_id=moderation_id)
        uid, order_id = moderation.uid, moderation.entity_id
        assert order_id is not None

        # Getting order with items
        order: Order = await self.storage.order.get(uid=uid, order_id=order_id)
        order.items = [item async for item in self.storage.item.get_for_order(uid=uid, order_id=order_id)]
        assert order.paymethod_id != PAYMETHOD_ID_OFFLINE

        return moderation, order

    async def handle(self) -> None:
        try:
            moderation, order = await self._fetch_entities(moderation_id=self.moderation_id)
            merchant = await GetMerchantAction(uid=moderation.uid, skip_data=True, skip_moderation=True).run()
        except StorageNotFound:
            self.logger.warning(f'Invalid moderation state for moderation {self.moderation_id}.')
            raise

        producer_cls = choose_moderation_producer_cls(order.fast_moderation)
        async with producer_cls(self.context.lb_factory) as producer:
            await producer.write_order(moderation=moderation, merchant=merchant, order=order)

        await self._log_moderation_scheduled(moderation=moderation, order=order)


class StartSubscriptionModerationWorkerAction(BaseDBAction):
    """Отправить модерацию подписки на обработку. Обработчик задач типа START_SUBSCRIPTION_MODERATION."""
    transact = True
    merchant: Merchant

    def __init__(self, moderation_id: int):
        super().__init__()
        self.moderation_id = moderation_id

    async def _fetch_entities(self, moderation_id: int) -> Tuple[Moderation, Subscription]:
        # Getting moderation
        moderation: Moderation = await self.storage.moderation.get(moderation_id=moderation_id)
        uid, subscription_id = moderation.uid, moderation.entity_id

        assert subscription_id is not None
        subscription: Subscription = await self.storage.subscription.get(uid=uid, subscription_id=subscription_id)

        self.merchant = await GetMerchantAction(uid=moderation.uid, skip_data=True, skip_moderation=True).run()

        return moderation, subscription

    async def _log_moderation_scheduled(self, moderation: Moderation, subscription: Subscription) -> None:
        assert subscription.revision is not None
        change_log = ChangeLog(
            uid=subscription.uid,
            revision=subscription.revision,
            operation=OperationKind.START_SUBSCRIPTION_MODERATION,
            info={'moderation_id': moderation.moderation_id},
        )
        await self.storage.change_log.create(change_log)

    async def _schedule_subscription_moderation(self, moderation_id: int) -> None:
        try:
            moderation, subscription = await self._fetch_entities(moderation_id=moderation_id)
        except StorageNotFound:
            self.logger.warning(f'Invalid moderation state for moderation {moderation_id}.')
            raise

        assert self.merchant is not None

        producer_cls = choose_moderation_producer_cls(subscription.fast_moderation)
        async with producer_cls(self.context.lb_factory) as producer:
            await producer.write_subscription(moderation=moderation,
                                              merchant=self.merchant,
                                              subscription=subscription)

        await self._log_moderation_scheduled(moderation=moderation, subscription=subscription)

    async def handle(self) -> None:
        await self._schedule_subscription_moderation(
            moderation_id=self.moderation_id,
        )


class UpdateModerationAction(APICallbackTaskMixin, BaseDBAction):
    """Given moderation result update corresponding Moderation entity.

    Moderation will be updated only if result is not outdated.
    We can receive several results - prefer the most recent result.
    To make result handling idempotent and sequential we:
        - select for update Moderation
        - update moderation only if moderation.unixtime < result.unixtime
    """
    transact = True

    def __init__(self, moderation_result: ModerationResult):
        super().__init__()
        self.moderation_result = moderation_result

    def _should_update(self, moderation: Moderation, moderation_result: ModerationResult) -> bool:
        return moderation.unixtime is None or moderation.unixtime < moderation_result.unixtime

    async def _update_moderation(self,
                                 moderation: Moderation,
                                 moderation_result: ModerationResult) -> Moderation:
        moderation.approved = moderation_result.approved
        moderation.unixtime = moderation_result.unixtime
        moderation.reason = moderation_result.reason
        moderation.reasons = moderation_result.reasons
        return await self.storage.moderation.save(moderation)

    def _collect_stats(self, moderation: Moderation) -> None:
        """
        Collect moderation resolving time (assuming moderation is resolved for the first time)
        """
        total_seconds = (moderation.updated - moderation.created).total_seconds()
        if moderation.moderation_type == ModerationType.MERCHANT:
            merchant_moderation_time.observe(total_seconds)
        elif moderation.moderation_type == ModerationType.ORDER:
            order_moderation_time.observe(total_seconds)
        elif moderation.moderation_type == ModerationType.SUBSCRIPTION:
            subscription_moderation_time.observe(total_seconds)

    async def _change_log_moderation_update(self, updated_moderation: Moderation) -> None:
        """Log moderation update into ChangeLog"""
        assert updated_moderation.revision is not None

        if updated_moderation.moderation_type == ModerationType.ORDER:
            operation = OperationKind.END_ORDER_MODERATION
        elif updated_moderation.moderation_type == ModerationType.MERCHANT:
            operation = OperationKind.END_MERCHANT_MODERATION
        elif updated_moderation.moderation_type == ModerationType.SUBSCRIPTION:
            operation = OperationKind.END_SUBSCRIPTION_MODERATION
        else:
            return

        await self.storage.change_log.create(ChangeLog(
            uid=updated_moderation.uid,
            revision=updated_moderation.revision,
            operation=operation,
            arguments={
                'moderation_id': updated_moderation.moderation_id,
                'approved': updated_moderation.approved,
                'reason': updated_moderation.reason,
                'reasons': updated_moderation.reasons,
            },
        ))

    async def _log_status_update(self, merchant: Merchant) -> None:
        # Load data not from the balance but from the db as we moderate the data from the db.
        merchant.load_data()
        log = MerchantModerationApprovedLog(
            merchant_uid=merchant.uid,
            merchant_name=merchant.name,
            merchant_acquirer=merchant.acquirer,
            merchant_full_name=merchant.organization.full_name,
            merchant_type=enum_value(merchant.organization.type),
            site_url=merchant.organization.site_url,
            moderation_received_at=utcnow(),
        )
        await self.pushers.log.push(log)

    async def _notify_about_merchant_moderation_result(self,
                                                       moderation: Moderation,
                                                       offer_external_id: Optional[str] = None) -> None:
        """Create notification tasks on result of Merchant moderation"""
        assert moderation.moderation_id is not None and moderation.unixtime is not None
        params = dict(
            moderation_id=moderation.moderation_id,
            unixtime=moderation.unixtime,
            offer_external_id=offer_external_id
        )
        task = Task(params=params, task_type=TaskType.MERCHANT_MODERATION_RESULT_NOTIFY)
        await self.storage.task.create(task)

    async def _on_merchant_moderation_updated(self, moderation: Moderation) -> None:
        if moderation.functionality_type == FunctionalityType.YANDEX_PAY:
            await self._on_yandex_pay_moderation_updated(moderation=moderation)
        elif moderation.functionality_type == FunctionalityType.PAYMENTS:
            await self._on_merchant_payments_moderation_updated(moderation=moderation)
        else:
            raise Exception('Unknown moderation functionality type')

    async def _on_yandex_pay_moderation_updated(self, moderation: Moderation) -> None:
        merchant = await GetMerchantAction(
            uid=moderation.uid,
            skip_moderation=True,
            skip_oauth=True
        ).run()

        async with self.file_storage.yandex_pay_admin.acquire() as s3:
            for document in merchant.documents:
                _, data_iter = await self.clients.payments_mds.download(document.path)
                async with s3.upload_stream(document.path) as uploader:
                    async for chunk in data_iter:
                        await uploader.write(chunk)

        await MerchantModerationResultNotifyAction(
            partner_id=merchant.functionalities.yandex_pay.partner_id,
            verified=moderation.approved,
            documents=merchant.documents,
        ).run()

    async def _on_merchant_payments_moderation_updated(self, moderation: Moderation) -> None:
        """Called when updated moderation has MERCHANT type"""
        offer_external_id = None
        if moderation.approved:
            merchant, offer_external_id = await InitContractAction(uid=moderation.uid).run()

            await InitProductsAction(uid=moderation.uid).run()
            async for service_merchant in self.storage.service_merchant.find(uid=moderation.uid, with_service=True):
                assert isinstance(service_merchant.service, Service)
                if service_fee := service_merchant.service.options.service_fee:
                    await InitProductsAction(uid=moderation.uid, service_fee=service_fee).run()

            await self._log_status_update(merchant)
        await self._notify_about_merchant_moderation_result(moderation=moderation, offer_external_id=offer_external_id)

        merchant = await GetMerchantAction(
            uid=moderation.uid,
            skip_data=True,
            skip_moderation=True,
            skip_oauth=True
        ).run()
        await self.create_merchant_moderation_updated_task(moderation, merchant=merchant)

        async for service_merchant in self.storage.service_merchant.find(uid=moderation.uid):
            service_id = service_merchant.service_id
            async for service_client in self.storage.service_client.find(service_id=service_id, with_service=True):
                service = cast(Service, service_client.service)
                service_client.service = NOT_FETCHED
                service.service_merchant = service_merchant
                service.service_client = service_client
                await self.create_merchant_moderation_updated_task(moderation, service=service)

    async def _on_moderation_updated(self, moderation: Moderation) -> None:
        """Called if moderation result is updated"""
        await self._change_log_moderation_update(updated_moderation=moderation)
        if moderation.ignore:
            return
        if moderation.moderation_type == ModerationType.MERCHANT:
            await self._on_merchant_moderation_updated(moderation=moderation)
        elif moderation.moderation_type == ModerationType.ORDER:
            if not moderation.approved:
                await NotifyAboutOrderModerationResultAction(order_moderation=moderation).run()
            try:
                assert moderation.entity_id
                tx = await self.storage.transaction.get_last_by_order(
                    uid=moderation.uid,
                    order_id=moderation.entity_id,
                    for_update=True,
                )
                assert tx is not None
                tx.reset_check_tries()
                await self.storage.transaction.save(tx)
            except TransactionNotFound:
                raise CoreFailError('Received moderation for an order without transaction')

    async def handle(self) -> Optional[Moderation]:
        moderation_result = self.moderation_result
        try:
            moderation = await self.storage.moderation.get(moderation_result.moderation_id, for_update=True)
        except ModerationNotFound:
            self.logger.context_push(moderation_id=moderation_result.moderation_id)
            self.logger.error('Moderation not found')
            return None

        if not self._should_update(moderation=moderation, moderation_result=moderation_result):
            return moderation

        first_result = moderation.approved is None
        moderation = await self._update_moderation(moderation=moderation, moderation_result=moderation_result)
        await self._on_moderation_updated(moderation=moderation)
        if first_result:
            self._collect_stats(moderation=moderation)
        return moderation


class NotifyAboutOrderModerationResultAction(BaseDBAction):
    """Create notifications when Order moderation result is received"""

    def __init__(self, order_moderation: Moderation):
        super().__init__()
        self.order_moderation: Moderation = order_moderation

    def email_render_context(self, moderation: Moderation, order: Order, merchant: Merchant) -> dict:
        assert order.items is not None

        return {
            'merchant': {
                'company_name': merchant.organization.full_name,
                'company_short_name': merchant.organization.name,
            },
            'order': {
                'order_id': order.order_id,
                'uid': order.uid,
                'caption': order.caption,
                'description': order.description,
            },
            'items': [
                {
                    'number': number,
                    'name': item.product.name,
                    'price': decimal_format(item.price),  # type: ignore
                    'amount': decimal_format(item.amount),
                    'total': decimal_format(item.total_price),  # type: ignore
                } for number, item in enumerate(order.items, 1) if item.product is not None
            ],
            'moderation': {
                'moderation_id': moderation.moderation_id,
                'reasons': moderation.reasons,
                'approved': moderation.approved,
                'unixtime': moderation.unixtime,
            }
        }

    async def notify_payer(self, moderation: Moderation, order: Order, merchant: Merchant) -> Task:
        """Create notification task to customer about order moderation result"""
        to_email = order.user_email
        mailing_id = settings.SENDER_MAILING_ORDER_MODERATION_NEGATIVE_FOR_CUSTOMER
        render_context = self.email_render_context(moderation=moderation, order=order, merchant=merchant)
        assert to_email is not None
        return await TransactEmailAction(
            to_email=to_email,
            mailing_id=mailing_id,
            render_context=render_context,
        ).run_async()

    async def notify_merchant(self, moderation: Moderation, order: Order, merchant: Merchant) -> Task:
        assert merchant.contact is not None
        to_email = merchant.contact.email
        mailing_id = settings.SENDER_MAILING_ORDER_MODERATION_NEGATIVE_FOR_MERCHANT
        render_context = self.email_render_context(moderation=moderation, order=order, merchant=merchant)
        return await TransactEmailAction(
            to_email=to_email,
            mailing_id=mailing_id,
            render_context=render_context,
        ).run_async()

    async def handle(self) -> None:
        moderation = self.order_moderation
        assert (moderation.moderation_type == ModerationType.ORDER
                and moderation.entity_id is not None
                and moderation.approved is not None)

        order = await GetOrderAction(uid=moderation.uid, order_id=moderation.entity_id, skip_add_crypto=True).run()
        merchant = await GetMerchantAction(uid=order.uid).run()

        await self.notify_payer(moderation=moderation, order=order, merchant=merchant)
        await self.notify_merchant(moderation=moderation, order=order, merchant=merchant)


class ProcessFastModerationRequestAction(BaseAction):
    def __init__(self, request: FastModerationRequest):
        super().__init__()
        self.request = request

    async def handle(self) -> None:
        async with FastModerationResponseProducer(self.context.lb_factory) as producer:
            await producer.write_dict({
                'service': self.request.service,
                'meta': self.request.meta,
                'type': self.request.type_,
                'unixtime': int(time.time() * 1000),
                'result': {
                    'verdict': 'Yes',
                },
            })
