from typing import Any, ClassVar, Collection, Dict, Optional

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.mixins.moderation import MerchantModerationMixin
from mail.payments.payments.core.entities.enums import FunctionalityType, ModerationType
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.moderation import Moderation
from mail.payments.payments.storage.exceptions import ModerationNotFound
from mail.payments.payments.utils.datetime import utcnow


class MerchantModerationResultNotifyWorkerAction(MerchantModerationMixin, BaseDBAction):
    """Обработчик задач на отправку писем о результате модерации продавца.
    По заданному ID модерации и временной метке проверяет, актуально ли уведомление, и если да,
    то отправляем письмо через клиент к сервису Рассылятор.
    """

    supported_functionality_types: ClassVar[Collection[FunctionalityType]] = (FunctionalityType.PAYMENTS,)

    def __init__(self, moderation_id: int, unixtime: int, offer_external_id: Optional[str] = None):
        super().__init__()
        self.moderation_id = moderation_id
        self.unixtime = unixtime
        self.offer_external_id = offer_external_id

    @staticmethod
    def get_mailing_id(moderation: Moderation) -> str:
        if moderation.approved:
            return settings.SENDER_MAILING_MERCHANT_MODERATION_APPROVED
        else:
            return settings.SENDER_MAILING_MERCHANT_MODERATION_DISAPPROVED

    async def send_merchant_moderation_result_letter(self,
                                                     moderation: Moderation,
                                                     merchant: Merchant,
                                                     offer_external_id: Optional[str] = None) -> str:
        """Отправить письмо о результате модерации продавца.
        В рассыляторе созданы транзакционные рассылки, от нас достаточно вызвать соответствующую ручку рассылки
        и предоставить минимально необходимый контекст для рендеринга шаблона письма.
        Returns:
            message_id: уникальный идентификатор письма
        """
        assert merchant.contact is not None
        to_email = merchant.contact.email
        render_context: Dict[str, Any] = {
            'offer_external_id': offer_external_id,
            'offer_id': merchant.contract_id,
            'reasons': moderation.reasons or None,
            'date': utcnow().strftime('%d.%m.%Y'),
        }

        return await self.clients.sender.send_transactional_letter(
            mailing_id=self.get_mailing_id(moderation),
            to_email=to_email,
            render_context=render_context,
        )

    async def handle(self) -> None:
        try:
            moderation = await self.storage.moderation.get(self.moderation_id, for_update=True)
        except ModerationNotFound:
            self.logger.context_push(moderation_id=self.moderation_id)
            self.logger.error('Moderation not found')
            return

        assert moderation.moderation_type == ModerationType.MERCHANT
        assert moderation.functionality_type is not None
        merchant: Merchant = await GetMerchantAction(uid=moderation.uid, skip_moderation=True).run()
        expired = (
            moderation.unixtime != self.unixtime
            or moderation.ignore
            or await self.has_ongoing_moderations(merchant, functionality_type=moderation.functionality_type)
        )

        if not expired and moderation.functionality_type in self.supported_functionality_types:
            self.logger.context_push(merchant_uid=merchant.uid,
                                     moderation_id=moderation.moderation_id,
                                     moderation_approved=moderation.approved)
            self.logger.info("Sending email notification about merchant moderation result")
            message_id = await self.send_merchant_moderation_result_letter(
                moderation=moderation,
                merchant=merchant,
                offer_external_id=self.offer_external_id,
            )
            self.logger.context_push(message_id=message_id, merchant_uid=merchant.uid)
            self.logger.info("Moderation result email notification is sent")
