from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get_acquirer import GetAcquirerMerchantAction
from mail.payments.payments.core.actions.order.send_to_history import SendToHistoryOrderAction
from mail.payments.payments.core.entities.enums import RefundStatus
from mail.payments.payments.interactions.trust.exceptions import TrustException


class StartRefundAction(BaseDBAction):
    def __init__(self, uid: int, order_id: int):
        super().__init__()
        self.uid = uid
        self.order_id = order_id

    async def handle(self) -> None:
        refund = await self.storage.order.get(self.uid, self.order_id)
        assert refund.shop is not None
        assert refund.refund_status is not None and refund.trust_refund_id is not None
        if refund.refund_status != RefundStatus.CREATED:
            raise Exception(f'Cannot start refund with status {refund.refund_status.value}')

        acquirer = await GetAcquirerMerchantAction(uid=refund.uid).run()
        acquirer = refund.get_acquirer(acquirer)

        try:
            trust = self.clients.get_trust_client(refund.uid, refund.shop.shop_type)
            await trust.refund_start(uid=refund.uid, acquirer=acquirer, refund_id=refund.trust_refund_id)
        except TrustException as e:
            self.logger.exception('Trust exception. Retrying task.')
            raise e
        else:
            refund.refund_status = RefundStatus.REQUESTED
            await self.storage.order.save(refund)
            assert refund.original_order_id is not None
            await SendToHistoryOrderAction(uid=refund.uid, order_id=refund.original_order_id).run_async()
