from __future__ import annotations

from enum import Enum, unique
from typing import ClassVar, Dict, FrozenSet, Optional, Set, cast

from sendr_taskqueue.worker.storage import TaskState, WorkerState  # noqa

from mail.payments.payments.utils.const import (
    SECONDS_IN_DAY, SECONDS_IN_MONTH, SECONDS_IN_SECOND, SECONDS_IN_WEEK, SECONDS_IN_YEAR
)

PAYMETHOD_ID_OFFLINE = 'offline'

PAY_METHOD_OFFLINE = 'offline'
PAY_METHOD_YANDEX = 'yandex'
PAY_METHODS = frozenset({PAY_METHOD_OFFLINE, PAY_METHOD_YANDEX})


@unique
class ArbitrageStatus(Enum):
    _ignore_ = ['ACTIVE_STATUSES']
    ACTIVE_STATUSES: ClassVar[FrozenSet['ArbitrageStatus']]

    CONSULTATION = 'consultation'
    ESCALATE = 'escalate'
    COMPLETE = 'complete'
    DISCARD = 'discard'


ArbitrageStatus.ACTIVE_STATUSES = frozenset({ArbitrageStatus.CONSULTATION, ArbitrageStatus.ESCALATE})


@unique
class ArbitrageVerdict(Enum):
    REFUND = 'refund'  # Вернуть средства
    DECLINE = 'decline'  # Отказать пользователю
    AGREE = 'agree'  # Стороны пришли к соглашению без арбитра


@unique
class DocumentType(Enum):
    OFFER = 'offer'
    OTHER = 'other'
    PASSPORT = 'passport'
    PROXY = 'proxy'  # Power of Attorney
    SIGNER_PASSPORT = 'signer_passport'
    PCI_DSS_CERT = 'pci_dss_cert'


@unique
class OperationKind(Enum):
    # Contract
    INIT_CONTRACT = 'init-contract'
    TERMINATE_CONTRACT = 'terminate-contract'

    # Merchant
    ADD_MERCHANT = 'add-merchant'
    EDIT_MERCHANT = 'edit-merchant'

    ADD_MERCHANT_DRAFT = 'add-merchant-draft'
    EDIT_MERCHANT_DRAFT = 'edit-merchant-draft'

    ADD_MERCHANT_PREREGISTRATION = 'add-merchant-preregistration'
    EDIT_MERCHANT_PREREGISTRATION = 'edit-merchant-preregistration'

    # Moderation
    START_MERCHANT_MODERATION = 'start-merchant-moderation'
    END_MERCHANT_MODERATION = 'end-merchant-moderation'
    START_ORDER_MODERATION = 'start-order-moderation'
    END_ORDER_MODERATION = 'end-order-moderation'
    START_SUBSCRIPTION_MODERATION = 'start-subscription-moderation'
    END_SUBSCRIPTION_MODERATION = 'end-subscription-moderation'

    # Order
    ADD_ORDER = 'add-order'
    ADD_REFUND = 'add-refund'
    UPDATE_ORDER = 'update-order'
    UPDATE_REFUND = 'update-refund'

    # Payment
    START_PAYMENT = 'start-payment'
    END_PAYMENT = 'end-payment'
    RECEIPT_CLOSE = 'receipt-close'

    # Transaction
    UPDATE_TX = 'update-tx'

    # Service_merchant
    ADD_SERVICE_MERCHANT = 'add-service-merchant'
    UPDATE_SERVICE_MERCHANT = 'update-service-merchant'
    DELETE_SERVICE_MERCHANT = 'delete-service-merchant'

    SEND_ORDER_EMAIL = 'send-order-email'


@unique
class PersonType(Enum):
    CEO = 'ceo'
    CONTACT = 'contact'
    SIGNER = 'signer'


@unique
class NDS(Enum):
    NDS_NONE = 'nds_none'
    NDS_0 = 'nds_0'
    NDS_10 = 'nds_10'
    NDS_10_110 = 'nds_10_110'
    NDS_18 = 'nds_18'
    NDS_18_118 = 'nds_18_118'
    NDS_20 = 'nds_20'
    NDS_20_120 = 'nds_20_120'

    @classmethod
    def to_arbitrage(cls, nds: 'NDS') -> str:
        mapping = {
            cls.NDS_NONE: 'VAT_0',
            cls.NDS_0: 'VAT_0',
            cls.NDS_10: 'VAT_10',
            cls.NDS_10_110: 'VAT_10',
            cls.NDS_18: 'VAT_20',
            cls.NDS_18_118: 'VAT_20',
            cls.NDS_20: 'VAT_20',
            cls.NDS_20_120: 'VAT_20',
        }
        return mapping[nds]


@unique
class TaskType(Enum):
    RUN_ACTION = 'run_action'
    API_CALLBACK = 'api_callback'
    START_MODERATION = 'start_moderation'
    START_REFUND = 'start_refund'
    START_ORDER_MODERATION = 'start_order_moderation'
    START_SUBSCRIPTION_MODERATION = 'start_subscription_moderation'
    MERCHANT_MODERATION_RESULT_NOTIFY = 'merchant_moderation_result_notify'
    UPDATE_REFUND = 'update_refund'


@unique
class CallbackMessageType(Enum):
    """
    Добавляет каплю семантики/контекста в параметры задачи (APICallbackTaskParams) на колбэк.
    Колбэк нужен при возникновении некоего события, и можно сказать, что так обозначается тип события.
    """
    CUSTOMER_SUBSCRIPTION_PROLONGATED = 'customer_subscription_prolongated'
    CUSTOMER_SUBSCRIPTION_STOPPED = 'customer_subscription_stopped'
    MERCHANT_MODERATION_STARTED = 'merchant_moderation_started'
    MERCHANT_MODERATION_UPDATED = 'merchant_moderation_updated'
    MERCHANT_REQUISITES_UPDATED = 'merchant_requisites_updated'
    ORDER_STATUS_UPDATED = 'order_status_updated'
    REFUND_STATUS_UPDATED = 'refund_status_updated'
    SERVICE_MERCHANT_UPDATED = 'service_merchant_updated'


@unique
class TransactionStatus(Enum):
    _ignore_ = ['FINAL_STATUSES', 'WAS_HELD_STATUSES', 'STATUS_TRANSITIONS']
    FINAL_STATUSES: ClassVar[FrozenSet['TransactionStatus']]
    WAS_HELD_STATUSES: ClassVar[FrozenSet['TransactionStatus']]
    STATUS_TRANSITIONS: ClassVar[Dict['TransactionStatus', FrozenSet['TransactionStatus']]]

    ACTIVE = 'active'
    FAILED = 'failed'
    CANCELLED = 'cancelled'
    HELD = 'held'
    CLEARED = 'cleared'

    @classmethod
    def was_held(cls, status: 'TransactionStatus') -> bool:
        return status in cls.WAS_HELD_STATUSES  # type: ignore

    @staticmethod
    def from_trust(trust_status: str) -> 'TransactionStatus':
        mapping = {
            'authorized': TransactionStatus.HELD,
            'not_authorized': TransactionStatus.FAILED,
            'cleared': TransactionStatus.CLEARED,
            'canceled': TransactionStatus.CANCELLED,
            'started': TransactionStatus.ACTIVE,
        }
        return mapping[trust_status]

    @staticmethod
    def to_trust(tx_status: 'TransactionStatus') -> str:
        mapping = {
            TransactionStatus.HELD: 'authorized',
            TransactionStatus.FAILED: 'not_authorized',
            TransactionStatus.CLEARED: 'cleared',
            TransactionStatus.CANCELLED: 'canceled',
            TransactionStatus.ACTIVE: 'started',
        }
        return mapping[tx_status]

    @property
    def possible_next_statuses(self) -> FrozenSet['TransactionStatus']:
        return self.STATUS_TRANSITIONS.get(self, frozenset())

    def allowed_next_status(self, new_status: TransactionStatus) -> bool:
        return new_status in self.possible_next_statuses

    @property
    def is_final(self) -> bool:
        return not bool(self.possible_next_statuses)


TransactionStatus.STATUS_TRANSITIONS = {
    TransactionStatus.ACTIVE: frozenset({
        TransactionStatus.CLEARED,
        TransactionStatus.FAILED,
        TransactionStatus.HELD,
    }),
    TransactionStatus.HELD: frozenset({
        TransactionStatus.CANCELLED,
        TransactionStatus.CLEARED,
    }),
}

TransactionStatus.WAS_HELD_STATUSES = frozenset({
    TransactionStatus.HELD,
    TransactionStatus.CANCELLED,
    TransactionStatus.CLEARED
})

TransactionStatus.FINAL_STATUSES = frozenset(
    status for status in list(TransactionStatus) if status.is_final
)


@unique
class OrderKind(Enum):
    PAY = 'pay'
    REFUND = 'refund'
    MULTI = 'multi'


@unique
class ProductStatus(Enum):
    ACTIVE = 'active'
    INACTIVE = 'inactive'


@unique
class MerchantStatus(Enum):
    NEW = 'new'
    ACTIVE = 'active'
    INACTIVE = 'inactive'
    DRAFT = 'draft'


@unique
class APICallbackSignMethod(Enum):
    ASYMMETRIC = 'asymmetric'
    JWT = 'jwt'


@unique
class OrderSource(Enum):
    UI = 'ui'
    SERVICE = 'service'
    SDK_API = 'sdk_api'


@unique
class MerchantType(Enum):
    OOO = 'ooo'
    IP = 'ip'


@unique
class RegistrationRoute(Enum):
    """
    Route meanings:
        OFFLINE: no acquirer required for merchant -> only offline orders awailable
        KASSA: merchant can have online orders and KASSA acquirer is used
        TINKOFF: merchant can have online orders and TINKOFF acquirer is used

    Tickets:
        PAYBACK-626
    """
    OFFLINE = 'offline'
    KASSA = 'kassa'
    TINKOFF = 'tinkoff'


@unique
class MerchantOAuthMode(Enum):
    PROD = 'prod'
    TEST = 'test'

    _from_shop_type: ClassVar[Dict['ShopType', 'MerchantOAuthMode']]
    _ignore_ = ['_from_shop_type']

    @classmethod
    def from_shop_type(cls, shop_type: 'ShopType') -> 'MerchantOAuthMode':
        return cast(Dict['ShopType', 'MerchantOAuthMode'], cls._from_shop_type)[shop_type]


@unique
class ShopType(Enum):
    PROD = 'prod'
    TEST = 'test'

    _from_oauth_mode: ClassVar[Dict['MerchantOAuthMode', 'ShopType']]
    _ignore_ = ['_from_oauth_mode']

    @classmethod
    def from_oauth_mode(cls, mode: 'MerchantOAuthMode') -> 'ShopType':
        return cast(Dict['MerchantOAuthMode', 'ShopType'], cls._from_oauth_mode)[mode]


MerchantOAuthMode._from_shop_type = {
    ShopType.PROD: MerchantOAuthMode.PROD,
    ShopType.TEST: MerchantOAuthMode.TEST,
}

ShopType._from_oauth_mode = {
    MerchantOAuthMode.PROD: ShopType.PROD,
    MerchantOAuthMode.TEST: ShopType.TEST,
}


@unique
class ModerationType(Enum):
    MERCHANT = 'merchant'
    ORDER = 'order'
    SUBSCRIPTION = 'subscription'


@unique
class ModerationStatus(Enum):  # front statuses
    NONE = 'none'
    ONGOING = 'ongoing'
    APPROVED = 'approved'
    REJECTED = 'rejected'


@unique
class ModerationRejectReason(Enum):
    _ignore_ = ['REASONS_MAPPING']
    REASONS_MAPPING: ClassVar[Dict[int, 'ModerationRejectReason']]
    DOCUMENT_PROBLEMS = 'Проблема с документами'
    UNKNOWN = 'Неизвестная причина'

    @classmethod
    def get_reason_str(cls, reason: Optional[int] = None) -> str:
        assert isinstance(cls.REASONS_MAPPING, dict)
        return cls.REASONS_MAPPING.get(reason, ModerationRejectReason.UNKNOWN).value


ModerationRejectReason.REASONS_MAPPING = {
    1: ModerationRejectReason.DOCUMENT_PROBLEMS
}


@unique
class PayStatus(Enum):
    _ignore_ = ['ALREADY_PAID_STATUSES', 'NON_TERMINAL_STATUSES']
    ALREADY_PAID_STATUSES: ClassVar[FrozenSet['PayStatus']]
    NON_TERMINAL_STATUSES: ClassVar[FrozenSet['PayStatus']]

    NEW = 'new'  # created

    IN_MODERATION = 'in_moderation'  # awaiting moderation result
    HELD = 'held'  # money on the order is frozen, awaiting manual clear or cancel
    IN_PROGRESS = 'in_progress'  # clear requested in trust
    PAID = 'paid'  # cleared

    REJECTED = 'rejected'  # transaction failed, can be paid
    MODERATION_NEGATIVE = 'moderation_negative'  # moderation failed, cannot be paid
    ABANDONED = 'abandoned'  # offline order, cancelled by timeout, cannot be paid

    IN_CANCEL = 'in_cancel'  # unhold requested in trust
    CANCELLED = 'cancelled'  # cancelled by merchant or was unholded, cannot be paid

    VERIFIED = 'verified'  # unused and deprecated

    @classmethod
    def is_already_paid(cls, pay_status: Optional['PayStatus']) -> bool:
        return pay_status in cast(Set, cls.ALREADY_PAID_STATUSES)


PayStatus.ALREADY_PAID_STATUSES = frozenset({PayStatus.PAID, PayStatus.IN_PROGRESS, PayStatus.IN_MODERATION})
PayStatus.NON_TERMINAL_STATUSES = frozenset({PayStatus.IN_CANCEL, PayStatus.IN_PROGRESS, PayStatus.IN_MODERATION})


@unique
class RefundStatus(Enum):
    CREATED = 'created'
    COMPLETED = 'completed'
    FAILED = 'failed'
    REQUESTED = 'requested'

    @staticmethod
    def from_trust(status):
        return {
            'success': RefundStatus.COMPLETED,
            'wait_for_notification': RefundStatus.REQUESTED,
        }.get(status, RefundStatus.FAILED)


@unique
class PeriodUnit(Enum):
    SECOND = 'S'
    DAY = 'D'
    WEEK = 'W'
    MONTH = 'M'
    YEAR = 'Y'

    @property
    def approx_seconds(self):
        seconds: Dict[PeriodUnit, int] = {
            self.SECOND: SECONDS_IN_SECOND,
            self.DAY: SECONDS_IN_DAY,
            self.WEEK: SECONDS_IN_WEEK,
            self.MONTH: SECONDS_IN_MONTH,
            self.YEAR: SECONDS_IN_YEAR,
        }

        assert self in seconds
        return seconds[self]


@unique
class WorkerType(Enum):
    ACTION_EXECUTOR = 'action_executor'
    CALLBACK_SENDER = 'callback_sender'
    MODERATION_READER = 'moderation_reader'
    MODERATION_WRITER = 'moderation_writer'
    REFUND_STARTER = 'refund_starter'
    REFUND_UPDATER = 'refund_updater'
    TRANSACTION_UPDATER = 'transaction_updater'
    ABANDON_TERMINATOR = 'abandon_terminator'
    CUSTOMER_SUBSCRIPTION_UPDATER = 'customer_subscription_updater'
    OAUTH_TOKEN_UPDATER = 'oauth_token_updater'
    MERCHANT_MODERATION_RESULT_NOTIFIER = 'merchant_moderation_result_notifier'
    MERCHANT_DATA_UPDATER = 'merchant_data_updater'
    FAST_MODERATION_PROCESSOR = 'fast_moderation_processor'


@unique
class Role(Enum):
    _ignore_ = ['_SUBROLES']
    _SUBROLES: Dict[Role, FrozenSet[Role]]
    _SUPERROLES: Dict[Role, FrozenSet[Role]]

    """Possible roles for Manager"""
    ACCOUNTANT = 'accountant'
    ADMIN = 'admin'
    ASSESSOR = 'assessor'

    @property
    def subroles(self) -> FrozenSet[Role]:
        return self._SUBROLES.get(self, frozenset())

    @property
    def superroles(self) -> FrozenSet[Role]:
        return self._SUPERROLES.get(self, frozenset())


Role._SUBROLES = {
    Role.ADMIN: frozenset({Role.ASSESSOR, Role.ACCOUNTANT}),
    Role.ACCOUNTANT: frozenset({Role.ASSESSOR}),
}
Role._SUPERROLES = {
    subrole: frozenset(
        superrole
        for superrole in Role
        if subrole in superrole.subroles
    )
    for subrole in Role
}


@unique
class PaymentsTestCase(Enum):
    """Possible test cases for orders"""
    TEST_OK_HELD = 'test_ok_held'
    TEST_OK_CLEAR = 'test_ok_clear'
    TEST_PAYMENT_FAILED = 'test_payment_failed'
    TEST_MODERATION_FAILED = 'test_moderation_failed'


@unique
class GroupType(Enum):
    DAY = 'day'
    WEEK = 'week'
    MONTH = 'month'


@unique
class GraphType(Enum):
    ALL_PAYMENTS_COUNT = 'all_payments_count'
    PAID_COUNT = 'paid_count'
    REFUND_COUNT = 'refund_count'
    REFUND_SUM = 'refund_sum'
    SUM_PAID_BILLS = 'sum_paid_bills'
    AVERAGE_BILL = 'average_bill'


@unique
class MerchantDraftPolicy(Enum):
    MERCHANT_DRAFT_ALLOWED = 'allowed'
    MERCHANT_DRAFT_REQUIRED = 'required'
    MERCHANT_DRAFT_FORBIDDEN = 'forbidden'


@unique
class MerchantRole(Enum):
    _ignore_ = ['_SUBROLES']
    _SUBROLES: Dict[MerchantRole, FrozenSet[MerchantRole]]
    _SUPERROLES: Dict[MerchantRole, FrozenSet[MerchantRole]]

    OWNER = 'owner'
    ADMIN = 'admin'
    OPERATOR = 'operator'
    VIEWER = 'viewer'

    @property
    def subroles(self) -> FrozenSet[MerchantRole]:
        return self._SUBROLES.get(self, frozenset())

    @property
    def superroles(self) -> FrozenSet[MerchantRole]:
        return self._SUPERROLES.get(self, frozenset())


MerchantRole._SUBROLES = {
    MerchantRole.OWNER: frozenset({MerchantRole.ADMIN, MerchantRole.OPERATOR, MerchantRole.VIEWER}),
    MerchantRole.ADMIN: frozenset({MerchantRole.OPERATOR, MerchantRole.VIEWER}),
    MerchantRole.OPERATOR: frozenset({MerchantRole.VIEWER}),
    MerchantRole.VIEWER: frozenset(),
}
MerchantRole._SUPERROLES = {
    subrole: frozenset(
        superrole
        for superrole in MerchantRole
        if subrole in superrole.subroles
    )
    for subrole in MerchantRole
}


@unique
class PaidOrderStatType(Enum):
    ORDER = 'order'
    SUBSCRIPTION = 'subscription'
    ORDER_FROM_MULTI_ORDER = 'order_from_multi_order'


@unique
class AcquirerType(Enum):
    TINKOFF = 'tinkoff'
    KASSA = 'kassa'


@unique
class TrustEnv(Enum):
    PROD = 'prod'
    SANDBOX = 'sandbox'


@unique
class ReceiptType(Enum):
    PREPAID = 'prepaid'
    COMPLETE = 'complete'


@unique
class CommonDataType(Enum):
    REGISTRATION_DATA = 'registration_data'


@unique
class OrderTimelineEventType(Enum):
    CREATED = 'created'
    PAID = 'paid'
    REFUNDED = 'refunded'
    PARTIALLY_REFUNDED = 'partially_refunded'
    PERIODIC_ACTIVE = 'periodic_active'
    PERIODIC_FAILED = 'periodic_failed'
    PERIODIC_CANCELLED = 'periodic_cancelled'
    PERIODIC_HELD = 'periodic_held'
    PERIODIC_CLEARED = 'periodic_cleared'
    PERIODIC_REFUNDED = 'periodic_refunded'

    @staticmethod
    def from_transaction_status(transaction_status: TransactionStatus) -> OrderTimelineEventType:
        mapping = {
            TransactionStatus.ACTIVE: OrderTimelineEventType.PERIODIC_ACTIVE,
            TransactionStatus.FAILED: OrderTimelineEventType.PERIODIC_FAILED,
            TransactionStatus.CANCELLED: OrderTimelineEventType.PERIODIC_CANCELLED,
            TransactionStatus.HELD: OrderTimelineEventType.PERIODIC_HELD,
            TransactionStatus.CLEARED: OrderTimelineEventType.PERIODIC_CLEARED,
        }
        return mapping[transaction_status]


@unique
class FunctionalityType(Enum):
    PAYMENTS = 'payments'
    YANDEX_PAY = 'yandex_pay'


@unique
class YandexPayPartnerType(Enum):
    MERCHANT = 'merchant'
    PAYMENT_GATEWAY = 'payment_gateway'


@unique
class YandexPayPaymentGatewayType(Enum):
    DIRECT_MERCHANT = 'direct_merchant'
    PSP = 'psp'
