from __future__ import annotations

from dataclasses import dataclass, field
from datetime import datetime
from typing import Optional, Protocol
from uuid import UUID

from sendr_aiopg import Entity
from sendr_utils.abc import abstractclass

from mail.payments.payments.core.entities.enums import (
    FunctionalityType, YandexPayPartnerType, YandexPayPaymentGatewayType
)
from mail.payments.payments.utils.datetime import utcnow


@dataclass
class FunctionalityData:
    @property
    def type(self) -> FunctionalityType:
        raise NotImplementedError


@dataclass
class PaymentsFunctionalityData(FunctionalityData):
    @property
    def type(self) -> FunctionalityType:
        return FunctionalityType.PAYMENTS


class PartnerIDProtocol(Protocol):
    partner_id: Optional[UUID]


@abstractclass
class YandexPayFunctionalityData(PartnerIDProtocol, FunctionalityData):
    @property
    def type(self) -> FunctionalityType:
        return FunctionalityType.YANDEX_PAY

    @property
    def partner_type(self) -> YandexPayPartnerType:
        raise NotImplementedError


@dataclass
class YandexPayPaymentGatewayFunctionalityData(YandexPayFunctionalityData):
    payment_gateway_type: YandexPayPaymentGatewayType
    gateway_id: str
    partner_id: Optional[UUID] = None

    @property
    def partner_type(self) -> YandexPayPartnerType:
        return YandexPayPartnerType.PAYMENT_GATEWAY


@dataclass
class YandexPayMerchantFunctionalityData(YandexPayFunctionalityData):
    merchant_gateway_id: Optional[str] = None
    merchant_desired_gateway: Optional[str] = None
    partner_id: Optional[UUID] = None

    @property
    def partner_type(self) -> YandexPayPartnerType:
        return YandexPayPartnerType.MERCHANT


@dataclass
class MerchantFunctionality(Entity):
    uid: int
    functionality_type: FunctionalityType
    data: FunctionalityData

    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)


@dataclass
class Functionalities:
    payments: Optional[PaymentsFunctionalityData] = None
    yandex_pay: Optional[YandexPayFunctionalityData] = None
