from dataclasses import asdict, dataclass
from decimal import Decimal
from typing import Optional

from mail.payments.payments.core.entities.enums import NDS
from mail.payments.payments.core.entities.image import Image
from mail.payments.payments.core.entities.product import Product


@dataclass
class Item:
    uid: int
    amount: Decimal
    product_id: Optional[int] = None
    order_id: Optional[int] = None
    new_price: Optional[Decimal] = None
    image_id: Optional[int] = None
    markup: Optional[dict] = None

    # Joined
    product: Optional[Product] = None
    image: Optional[Image] = None
    trust_order_id: Optional[str] = None

    @property
    def currency(self) -> Optional[str]:
        return getattr(self.product, 'currency', None)

    @property
    def name(self) -> Optional[str]:
        return getattr(self.product, 'name', None)

    @property
    def nds(self) -> Optional[NDS]:
        return getattr(self.product, 'nds', None)

    @property
    def price(self) -> Optional[Decimal]:
        if self.new_price is not None:
            return self.new_price
        return getattr(self.product, 'price', None)

    @property
    def total_price(self) -> Optional[Decimal]:
        if self.amount is not None and self.price is not None:
            return self.amount * self.price
        return None

    def dump(self) -> dict:
        data = asdict(self)
        data['amount'] = float(round(self.amount, 2))
        if self.new_price is not None:
            data['new_price'] = float(round(self.new_price, 2))
        data.pop('product')
        if self.product is not None:
            data.update({
                'currency': self.product.currency,
                'name': self.product.name,
                'nds': self.product.nds.value,
                'price': float(round(self.product.price, 2))
            })
        return data
