from dataclasses import dataclass, fields
from datetime import datetime
from decimal import Decimal
from typing import Any, Generic, List, Literal, Optional, Tuple, TypeVar

from mail.payments.payments.core.exceptions import KeysetInvalidError

KeysetSortOrder = Literal['asc', 'desc']
KeysetColumn = str
KeysetRestriction = Any  # Comparable
Keyset = List[Tuple[KeysetColumn, KeysetSortOrder, KeysetRestriction]]

ValueType = TypeVar('ValueType')


@dataclass
class KeysetEntry(Generic[ValueType]):
    """
    Граничное значение поля для последующей фильтрации объектов по этому полю.

    :param barrier: граничное значение, грань
    :param order: направление сортировки поля, определяет направление сравнения с гранью.
        `'asc'` - нужны записи со значениями поля больше грани (нижняя грань / lower bound)
        `'desc'` - нужны записи со значениями поля меньше грани (верхняя рань / upper bound)
    """
    barrier: ValueType
    order: KeysetSortOrder


@dataclass
class BaseKeysetEntity:
    """
    Кийсет хранит в себе информации о сортировке некоторой коллекции объектов по нескольким полям.
    Эта информация включает в себя:
        - приоритет сортировки полей
        - порядок сортировки каждого поля (возрастание / убывание)
        - граничное значение поля для определения "хвоста" коллекции в соответствии с параметрами сортировки

    Например, пусть кийсет хранит в себе, что коллекция объектов отсортирована по полям
    `a` (по возрастание), `b` (по убыванию), `c` (по возрастанию).
    Граничные значения полей соответственно равны A, B, C.

    Тогда извлечении объектов по заданному кийсету будет добавлено условие сортировки вида:
        SELECT *
        FROM objects
        ORDER BY a ASC, b DESC, c ASC

    и условие границы:
        WHERE
            -- записи "большие" только по полю c
            (a = A) AND (b = B) AND (C > c)
        OR
            -- записи "большие" только по полю b
            (a = A) AND (b < B)
        OR
            -- записи "большие" только по полю a
            (a > A)

    :param sort_order: приоритет сортировки полей в виде списка строк-имен полей от старшего ко младшему
    """

    sort_order: List[str]

    def __post_init__(self) -> None:
        acceptable_fields = set(field.name for field in fields(self))
        acceptable_fields.discard('sort_order')
        for name in self.sort_order:
            if name not in acceptable_fields:
                raise KeysetInvalidError(params={'description': 'Not acceptable key set field', 'name': name})
            if getattr(self, name) is None:
                raise KeysetInvalidError(params={'description': 'Key set field has bad value', 'name': name})


@dataclass
class ManagerMerchantListKeysetEntity(BaseKeysetEntity):
    created: Optional[KeysetEntry[datetime]] = None
    updated: Optional[KeysetEntry[datetime]] = None
    uid: Optional[KeysetEntry[int]] = None


@dataclass
class ManagerAnalyticsKeysetEntity(BaseKeysetEntity):
    uid: Optional[KeysetEntry[int]] = None
    created: Optional[KeysetEntry[datetime]] = None
    payments_success: Optional[KeysetEntry[int]] = None
    money_success: Optional[KeysetEntry[Decimal]] = None
