from dataclasses import dataclass
from datetime import datetime
from typing import List, Optional

from sendr_writers.base.pusher import BaseLog

from .enums import AcquirerType, OrderSource


@dataclass
class MerchantShortInfo:
    merchant_uid: int
    merchant_name: Optional[str]
    merchant_acquirer: Optional[AcquirerType]


@dataclass
class OrderCreatedLog(MerchantShortInfo, BaseLog):
    MESSAGE = 'ORDER_CREATED'

    order_id: int
    price: float
    items: List[dict]
    sdk_api_created: bool
    sdk_api_pay: bool
    created_by_source: Optional[OrderSource]
    pay_by_source: Optional[OrderSource]
    kind: Optional[str] = None
    status: Optional[str] = None
    customer_subscription_id: Optional[int] = None
    customer_uid: Optional[int] = None
    service_id: Optional[int] = None
    service_name: Optional[str] = None
    merchant_oauth_mode: Optional[str] = None


@dataclass
class OrderUpdatedLog(OrderCreatedLog):
    MESSAGE = 'ORDER_UPDATED'


@dataclass
class OrderStatusUpdatedLog(MerchantShortInfo, BaseLog):
    MESSAGE = 'ORDER_STATUS_UPDATED'

    order_id: int
    purchase_token: Optional[str]
    status: str
    sdk_api_created: bool
    sdk_api_pay: bool
    price: float
    created_by_source: Optional[OrderSource]
    pay_by_source: Optional[OrderSource]
    customer_uid: Optional[int] = None
    service_id: Optional[int] = None
    service_name: Optional[str] = None
    paymethod_id: Optional[str] = None
    referer: Optional[str] = None
    user_agent: Optional[str] = None
    merchant_oauth_mode: Optional[str] = None


@dataclass
class OrderPaymentStartedLog(MerchantShortInfo, BaseLog):
    MESSAGE = 'ORDER_PAYMENT_STARTED'

    order_id: int
    purchase_token: str
    sdk_api_created: bool
    sdk_api_pay: bool
    price: float
    created_by_source: Optional[OrderSource]
    pay_by_source: Optional[OrderSource]
    pay_token: Optional[str] = None
    referer: Optional[str] = None
    user_agent: Optional[str] = None
    customer_uid: Optional[int] = None
    service_id: Optional[int] = None
    service_name: Optional[str] = None
    merchant_oauth_mode: Optional[str] = None
    turboapp_id: Optional[str] = None
    tsid: Optional[str] = None
    psuid: Optional[str] = None


@dataclass
class RefundCreatedLog(MerchantShortInfo, BaseLog):
    MESSAGE = 'REFUND_CREATED'

    refund_id: int
    order_id: int
    status: str
    price: float
    items: List[dict]
    paymethod_id: Optional[str] = None


@dataclass
class RefundStatusUpdatedLog(MerchantShortInfo, BaseLog):
    MESSAGE = 'REFUND_STATUS_UPDATED'

    refund_id: int
    order_id: int
    status: str
    price: float


@dataclass
class MerchantModerationApprovedLog(MerchantShortInfo, BaseLog):
    MESSAGE = 'MODERATION_APPROVED'

    moderation_received_at: datetime
    merchant_full_name: Optional[str]
    merchant_type: Optional[str]
    site_url: Optional[str] = None


@dataclass
class ServiceMerchantCreatedLog(BaseLog):
    MESSAGE = 'SERVICE_MERCHANT_CREATED'

    merchant_uid: int
    service_id: int
    service_name: str
    service_merchant_id: int
    enabled: bool


@dataclass
class ServiceMerchantUpdatedLog(ServiceMerchantCreatedLog):
    MESSAGE = 'SERVICE_MERCHANT_UPDATED'
