from __future__ import annotations

from collections import defaultdict
from dataclasses import dataclass, field, fields
from datetime import date, datetime
from decimal import Decimal
from typing import Any, ClassVar, Dict, List, Mapping, Optional, Tuple, Union

from sendr_utils import enum_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.common import SearchStats
from mail.payments.payments.core.entities.document import Document
from mail.payments.payments.core.entities.enums import (
    AcquirerType, APICallbackSignMethod, DocumentType, FunctionalityType, MerchantStatus, MerchantType, OrderKind,
    OrderSource, PersonType, RegistrationRoute
)
from mail.payments.payments.core.entities.functionality import Functionalities
from mail.payments.payments.core.entities.keyset import ManagerAnalyticsKeysetEntity, ManagerMerchantListKeysetEntity
from mail.payments.payments.core.entities.merchant_oauth import MerchantOAuth
from mail.payments.payments.core.entities.merchant_preregistration import MerchantPreregistration, PreregisterData
from mail.payments.payments.core.entities.moderation import ModerationData
from mail.payments.payments.core.entities.not_fetched import NOT_FETCHED, NotFetchedType
from mail.payments.payments.core.entities.service import OfferSettings, ServiceMerchant
from mail.payments.payments.core.exceptions import TinkoffInvalidSubmerchantIdError, UnknownAcquirerType
from mail.payments.payments.utils.datetime import utcnow


@dataclass
class AddressData:
    type: str
    city: str
    country: str
    street: str
    zip: str
    home: Optional[str] = None


@dataclass
class BankData:
    account: str
    bik: str
    correspondent_account: Optional[str] = None
    name: Optional[str] = None


@dataclass
class PersonData:
    type: PersonType
    name: str
    email: str
    phone: str
    surname: str
    patronymic: Optional[str] = None
    birth_date: Optional[date] = None


@dataclass
class OrganizationData:
    type: Optional[MerchantType] = None
    name: Optional[str] = None
    english_name: Optional[str] = None
    full_name: Optional[str] = None
    inn: Optional[str] = None
    kpp: Optional[str] = None
    ogrn: Optional[str] = None
    schedule_text: Optional[str] = None
    site_url: Optional[str] = None
    description: Optional[str] = None


@dataclass
class MerchantData:
    addresses: Optional[List[AddressData]] = None
    bank: Optional[BankData] = None
    organization: Optional[OrganizationData] = None
    persons: Optional[List[PersonData]] = None
    username: Optional[str] = None
    registered: bool = True
    fast_moderation: bool = False

    def get_person(self, type_: PersonType) -> Optional[PersonData]:
        if self.persons:
            return next((p for p in self.persons if p.type == type_), None)
        return None


@dataclass
class APICallbackParams:
    sign_method: APICallbackSignMethod = APICallbackSignMethod.ASYMMETRIC
    secret: Optional[str] = None


@dataclass
class PaymentSystemsOptions:
    apple_pay_enabled: bool = settings.MERCHANT_PAYSYS_ENABLED_APPLE_PAY_DEFAULT
    google_pay_enabled: bool = settings.MERCHANT_PAYSYS_ENABLED_GOOGLE_PAY_DEFAULT


@dataclass
class MerchantOptions:
    offer_settings: OfferSettings = field(default_factory=OfferSettings)
    order_offline_abandon_period: int = settings.ORDER_OFFLINE_ABANDON_PERIOD_DEFAULT
    allow_create_service_merchants: bool = True
    allowed_order_sources: Optional[List[OrderSource]] = None
    hide_commission: bool = False
    can_skip_registration: bool = False
    payment_systems: PaymentSystemsOptions = field(default_factory=PaymentSystemsOptions)
    order_max_total_price: Optional[int] = None  # https://st.yandex-team.ru/PAYBACK-915

    def is_creating_order_allowed(self, source: OrderSource) -> bool:
        if (allowed_order_sources := self.allowed_order_sources) is not None:
            return source in allowed_order_sources
        return True

    @property
    def allow_create_orders_via_ui(self):
        return self.is_creating_order_allowed(OrderSource.UI)


@dataclass(eq=False)
class Merchant:
    """Represents merchant in database.
    Billing fields require parent being loaded.
    To access data fields, properties should be used, since they enforce data being loaded.
    """

    uid: int
    name: Optional[str] = None
    merchant_id: Optional[str] = None

    trustworthy: bool = False
    revision: int = 0
    status: MerchantStatus = MerchantStatus.NEW
    blocked: bool = False
    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)

    # Billing fields
    client_id: Optional[str] = None
    person_id: Optional[str] = None
    contract_id: Optional[str] = None
    submerchant_id: Optional[str] = None

    # Data
    data: Optional[MerchantData] = None
    data_updated_at: datetime = field(default_factory=utcnow)
    data_locked: bool = False

    documents: List[Document] = field(default_factory=list)
    parent_uid: Optional[int] = None
    api_callback_url: Optional[str] = None
    api_callback_params: APICallbackParams = field(default_factory=APICallbackParams)

    token: Optional[str] = None
    acquirer: Optional[AcquirerType] = AcquirerType(settings.DEFAULT_ACQUIRER)

    dialogs_org_id: Optional[str] = None
    support_comment: Optional[str] = None
    options: MerchantOptions = field(default_factory=MerchantOptions)

    # Generated
    moderation: Optional[ModerationData] = None
    moderations: Optional[Dict[FunctionalityType, ModerationData]] = None
    draft_data: Optional[dict] = None

    # Joined
    preregistration: Optional[MerchantPreregistration] = None
    parent: Optional['Merchant'] = None
    oauth: Union[NotFetchedType, List[MerchantOAuth]] = NOT_FETCHED
    functionalities: Union[NotFetchedType, Functionalities] = NOT_FETCHED

    _PARENT_FIELDS: ClassVar[Tuple[str, ...]] = (
        'client_id',
        'person_id',
        'contract_id',
        'submerchant_id',
        'acquirer',
    )
    _DATA_FIELDS: ClassVar[Tuple[str, ...]] = tuple((f.name for f in fields(MerchantData)))
    _parent_loaded: bool = False
    _data_loaded: bool = False

    def __post_init__(self):
        if self.merchant_id is None:
            self.merchant_id = str(self.uid)
        if self.parent_uid is None:
            self._parent_loaded = True

    def __eq__(self, other: Any) -> bool:
        return isinstance(other, Merchant) and all((
            getattr(self, field.name) == getattr(other, field.name)
            for field in fields(self)
            if field.name[0] != '_'
        ))

    def __getattribute__(self, name: str) -> Any:
        if name not in ('_PARENT_FIELDS', '_DATA_FIELDS', '_parent_loaded', '_data_loaded'):
            if name in self._PARENT_FIELDS and not self._parent_loaded:
                raise RuntimeError(f'Cannot access {name}. Merchant parent must be loaded first')
            if name in self._DATA_FIELDS and not self._data_loaded:
                raise RuntimeError(f'Cannot access {name}. Merchant data must be loaded first')
        return super().__getattribute__(name)

    @property
    def registration_route(self) -> RegistrationRoute:
        if self.acquirer is None:
            return RegistrationRoute.OFFLINE
        elif self.acquirer == AcquirerType.TINKOFF:
            return RegistrationRoute.TINKOFF
        elif self.acquirer == AcquirerType.KASSA:
            return RegistrationRoute.KASSA

        raise UnknownAcquirerType(
            params=dict(uid=self.uid, acquirer=enum_value(self.acquirer))
        )

    @property
    def preregister_data(self) -> Optional[PreregisterData]:
        # чтобы отдавать в АПИ в схеме Мерчанта
        assert self.preregistration is not None
        return self.preregistration.data.preregister_data

    @property
    def addresses(self) -> Optional[List[AddressData]]:
        assert self.data is not None
        return self.data.addresses

    @property
    def bank(self) -> Optional[BankData]:
        assert self.data is not None
        return self.data.bank

    @property
    def ceo(self) -> Optional[PersonData]:
        if getattr(self, '_ceo', None) is None:
            self._ceo = self.get_person(PersonType.CEO)
        return self._ceo

    @property
    def contact(self) -> Optional[PersonData]:
        if getattr(self, '_contact', None) is None:
            self._contact = self.get_person(PersonType.CONTACT) or self.ceo
        return self._contact

    @property
    def persons(self) -> Optional[List[PersonData]]:
        assert self.data is not None
        return self.data.persons

    @property
    def organization(self) -> OrganizationData:
        assert self.data is not None and self.data.organization is not None
        return self.data.organization

    @property
    def username(self) -> Optional[str]:
        assert self.data is not None
        return self.data.username

    @property
    def registered(self) -> bool:
        assert self.data is not None
        return self.data.registered

    @property
    def fast_moderation(self) -> bool:
        assert self.data is not None
        return self.data.fast_moderation

    @property
    def data_loaded(self) -> bool:
        return self._data_loaded

    @property
    def parent_loaded(self) -> bool:
        return self._parent_loaded

    def get_address_by_type(self, type_: str) -> Optional[AddressData]:
        if self.addresses is None:
            return None
        return next((a for a in self.addresses if a.type == type_), None)

    def get_document(self, path: str) -> Optional[Document]:
        return next((d for d in self.documents if d.path == path), None)

    def get_documents_by_type(self) -> Mapping[DocumentType, List[Document]]:
        result: dict = defaultdict(list)
        for document in self.documents:
            result[document.document_type].append(document)
        return result

    def get_person(self, type_: PersonType) -> Optional[PersonData]:
        assert self._data_loaded and self.data
        return self.data.get_person(type_)

    def load_parent(self):
        if self.parent_uid:
            assert self.parent is not None
            self.client_id = self.parent.client_id
            self.person_id = self.parent.person_id
            self.contract_id = self.parent.contract_id
            self.submerchant_id = self.parent.submerchant_id
            self.acquirer = self.parent.acquirer
            self.data = self.parent.data
        self._parent_loaded = True

    def load_data(self):
        self._data_loaded = True

    def get_submerchant_id(self) -> Optional[str]:
        if self.acquirer == AcquirerType.TINKOFF and not self.submerchant_id:
            raise TinkoffInvalidSubmerchantIdError
        return self.submerchant_id


@dataclass
class MerchantStat:
    parent_uid: Optional[str] = None
    name: Optional[str] = None
    orders_sum: Decimal = field(default_factory=Decimal)
    commission: Decimal = field(default_factory=Decimal)
    orders_kind: Optional[OrderKind] = None
    orders_paid_count: int = 0
    orders_created_count: int = 0

    @property
    def money_average(self) -> Decimal:
        if self.orders_paid_count == 0:
            return Decimal(0)
        return self.orders_sum / Decimal(self.orders_paid_count)


@dataclass
class MerchantAnalyticsStats:
    payments_total: int = 0
    payments_success: int = 0
    payments_refund: int = 0
    money_success: Optional[Decimal] = None
    money_refund: Optional[Decimal] = None

    def without_money(self) -> MerchantAnalyticsStats:
        return MerchantAnalyticsStats(
            payments_total=self.payments_total,
            payments_success=self.payments_success,
            payments_refund=self.payments_refund,
        )


@dataclass
class AllPaymentsPoint:
    x: datetime
    y: float


@dataclass
class MerchantsAdminData:
    merchants: List[Merchant]
    stats: SearchStats
    keyset: Optional[ManagerMerchantListKeysetEntity]


@dataclass
class MerchantWithAnalytics:
    uid: int
    name: Optional[str]
    moderation: Optional[ModerationData]
    created: datetime
    acquirer: Optional[AcquirerType]
    blocked: bool
    client_id: Optional[str]
    organization: Optional[OrganizationData]
    service_merchants: List[ServiceMerchant]
    support_comment: Optional[str]
    contact: Optional[PersonData]

    payments_total: int
    payments_success: int
    payments_refund: int
    money_success: Optional[Decimal]
    money_refund: Optional[Decimal]

    @property
    def money_aov(self) -> Decimal:
        if self.money_success is None or self.payments_success == 0:
            return Decimal('0')
        return self.money_success / self.payments_success


@dataclass
class MerchantsAnalyticsAdminData:
    merchants: List[MerchantWithAnalytics]
    stats: SearchStats
    keyset: Optional[ManagerAnalyticsKeysetEntity]
