from dataclasses import dataclass, field
from datetime import datetime
from typing import ClassVar, Optional, Type

from sendr_aiopg import Entity
from sendr_utils import utcnow

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import MerchantOAuthMode
from mail.payments.payments.utils.crypto import decrypt, encrypt


@dataclass
class MerchantOAuth(Entity):
    DoesNotExist: ClassVar[Type[Exception]]

    uid: int
    shop_id: int
    expires: datetime

    mode: MerchantOAuthMode = MerchantOAuthMode.PROD
    poll: bool = True
    data: dict = field(default_factory=dict)
    encrypted_access_token: Optional[str] = None
    encrypted_refresh_token: Optional[str] = None
    key_version: int = settings.ENCRYPTION_CURRENT_VERSION
    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)

    @property
    def decrypted_access_token(self) -> str:
        assert self.encrypted_access_token
        return decrypt(self.encrypted_access_token, self.key_version)

    @decrypted_access_token.setter
    def decrypted_access_token(self, value: str) -> None:
        self.encrypted_access_token = encrypt(value, self.key_version)

    @property
    def decrypted_refresh_token(self) -> str:
        assert self.encrypted_refresh_token
        return decrypt(self.encrypted_refresh_token, self.key_version)

    @decrypted_refresh_token.setter
    def decrypted_refresh_token(self, value: str) -> None:
        self.encrypted_refresh_token = encrypt(value, self.key_version)

    @property
    def expires_in(self) -> float:
        return (self.expires - utcnow()).total_seconds()

    @property
    def expired(self) -> bool:
        return self.expires <= utcnow()


@dataclass
class OAuthToken:
    token_type: str
    access_token: str
    refresh_token: str
    expires_in: int
