from dataclasses import dataclass, field
from datetime import datetime
from typing import List, Optional

from mail.payments.payments.utils.datetime import utcnow


@dataclass
class PreregisterData:
    inn: Optional[str]
    services: List[int]
    categories: List[int]
    require_online: bool


@dataclass
class MerchantPreregistrationData:
    """
    :ivar preregister_data:
        Данные пререгистрации мерчанта, прошедшие валидацию и по которым была создан мерчант.
    :ivar raw_preregister_data:
        Данные пререгистрации мерчанта, не прошедшие валидацию (хотя и корректные по схеме).
        Храним в базе в сущности регистрации, чтобы помнить попытку зарегистрироваться,
        если таковая оказалась невозможной по текущим правилам валидации данных, т. е. мерчант
        не был создан в базе - будет только запись о регистрации.
    """
    raw_preregister_data: Optional[PreregisterData] = None
    preregister_data: Optional[PreregisterData] = None


@dataclass
class MerchantPreregistration:
    uid: int
    data: MerchantPreregistrationData
    created: datetime = field(default_factory=utcnow, compare=False)
    updated: datetime = field(default_factory=utcnow, compare=False)

    @property
    def preregister_data(self) -> Optional[PreregisterData]:
        return self.data.preregister_data

    @property
    def raw_preregister_data(self) -> Optional[PreregisterData]:
        return self.data.raw_preregister_data
