from dataclasses import dataclass, field
from datetime import datetime
from typing import List, Optional

from mail.payments.payments.core.entities.enums import FunctionalityType, ModerationStatus, ModerationType
from mail.payments.payments.utils.datetime import utcnow


@dataclass
class Moderation:
    """Факт модерации сущности, нуждающейся в модерации.

    Сейчас модерацию должны проходить продавцы и, возможно, заказы (в момент их оплаты).
    Хотя модерацию заказов называют "модерацией транзакций", нужно проверить состав заказа.
    Необходимость проверки заказа возникает в момент оплаты заказа покупателем, отсюда возникает путаница.
    В будущем, возможно, модерация заказов будет заменена модерацией продуктов,
    а модерация заказов будет не нужна. Однако поскольку сейчас продукты создаются для каждого
    заказа вновь и вновь - приходится делать модерацию заказов, чтобы не брать деньги за что-то незаконное.

    Для продавцов может быть несколько модераций.
    Для заказа мы считаем, что может быть одна модерация.

    Fields:
        unixtime: метка времени в виде целого числа секунд - приходит из ответа модерации, так мы помечаем
            результат модерации, чтобы не перезаписать его более старым
            можно утверждать, что (unixtime is None) <=> (approved is None) - а если не так, то баг
    """

    uid: int
    moderation_type: ModerationType
    functionality_type: Optional[FunctionalityType] = None
    entity_id: Optional[int] = None
    revision: Optional[int] = None
    moderation_id: Optional[int] = None
    approved: Optional[bool] = None
    unixtime: Optional[int] = None
    reasons: List[int] = field(default_factory=list)
    reason: Optional[str] = None
    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)
    ignore: bool = False


@dataclass
class ModerationResult:
    """Результат модерации в виде сущности - создается из приходящих извне ответов.
    Эфемерен - не хранится в базе. Используется в моменты установления результата на модерации.
    """
    moderation_id: int
    client_id: int
    approved: bool
    unixtime: int
    submerchant_id: Optional[int]
    reason: Optional[str]  # todo: legacy поле - как перейдем на reasons, нужно отовсюду выпилить reason
    reasons: List[int] = field(default_factory=list)


@dataclass
class ModerationData:
    approved: Optional[bool] = None
    has_moderation: Optional[bool] = None
    has_ongoing: Optional[bool] = None
    reasons: List[int] = field(default_factory=list)
    updated: Optional[datetime] = None

    @property
    def status(self) -> ModerationStatus:
        if not self.has_moderation:
            return ModerationStatus.NONE
        if self.approved:
            return ModerationStatus.APPROVED
        if self.has_ongoing:
            return ModerationStatus.ONGOING
        return ModerationStatus.REJECTED


@dataclass
class FastModerationRequest:
    service: str
    type_: str
    meta: dict
    unixtime: int
