from dataclasses import dataclass, field
from datetime import datetime
from decimal import Decimal
from typing import Any, Dict, List, Optional

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.arbitrage import Arbitrage
from mail.payments.payments.core.entities.common import SearchStats
from mail.payments.payments.core.entities.customer_subscription import CustomerSubscription
from mail.payments.payments.core.entities.enums import (
    PAY_METHOD_OFFLINE, PAY_METHOD_YANDEX, PAYMETHOD_ID_OFFLINE, AcquirerType, MerchantOAuthMode, OrderKind,
    OrderSource, OrderTimelineEventType, PaymentsTestCase, PayStatus, ReceiptType, RefundStatus, ShopType
)
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.service import ServiceMerchant
from mail.payments.payments.core.entities.shop import Shop
from mail.payments.payments.http_helpers.crypto import Crypto
from mail.payments.payments.utils.datetime import utcnow


@dataclass
class OrderData:
    """
    turboapp_id:
        TurboApp identifier
        See: PAYMENTSDK-137
    tsid
        TurboApp Session ID, tsid is generated on TurboApp is open.
        tsid set if order payment started through TurboApp inside SuperApp
        See: PAYMENTSDK-137
    psuid:
        Publisher Specific User Identifier
        tsid set if order payment started through TurboApp inside SuperApp
        See: PAYMENTSDK-137
    """
    trust_form_name: Optional[str] = None
    trust_template: Optional[str] = None
    multi_max_amount: Optional[int] = None
    multi_issued: int = 0
    meta: Optional[str] = None
    offline_prolongation_amount: Optional[int] = None
    service_data: Optional[dict] = None
    turboapp_id: Optional[str] = None
    tsid: Optional[str] = None
    psuid: Optional[str] = None
    receipt_type: ReceiptType = ReceiptType.COMPLETE
    fast_moderation: bool = False
    recurrent: bool = False
    without_3ds: bool = False
    version: int = 2


@dataclass
class OriginalOrderInfo:
    pay_status: Optional[PayStatus] = PayStatus.NEW
    paymethod_id: Optional[str] = None

    @property
    def is_already_paid(self) -> bool:
        return PayStatus.is_already_paid(self.pay_status)

    @property
    def pay_method(self) -> Optional[str]:
        if not self.is_already_paid:
            return None
        elif self.paymethod_id == PAYMETHOD_ID_OFFLINE:
            return 'offline'
        else:
            return 'yandex'


@dataclass
class OrderTimelineEvent:
    date: datetime
    event_type: OrderTimelineEventType
    extra: Dict[str, Any] = field(default_factory=dict)


@dataclass
class Order:
    uid: int
    shop_id: int
    order_id: Optional[int] = None
    parent_order_id: Optional[int] = None
    original_order_id: Optional[int] = None
    revision: Optional[int] = None
    acquirer: Optional[AcquirerType] = None  # https://st.yandex-team.ru/PAYBACK-584
    commission: Optional[int] = None  # PAYBACK-842 в базисных пунктах (1% == 100 б.п.)

    kind: OrderKind = OrderKind.PAY
    pay_status: Optional[PayStatus] = PayStatus.NEW
    refund_status: Optional[RefundStatus] = None

    active: bool = True
    autoclear: bool = True
    verified: bool = False  # currently unused and probably deprecated
    created_by_source: OrderSource = OrderSource.UI
    pay_by_source: OrderSource = OrderSource.UI

    closed: Optional[datetime] = None
    created: Optional[datetime] = None
    updated: Optional[datetime] = None
    held_at: Optional[datetime] = None
    pay_status_updated_at: Optional[datetime] = None

    caption: Optional[str] = None
    description: Optional[str] = None
    user_email: Optional[str] = None
    user_description: Optional[str] = None
    trust_refund_id: Optional[str] = None

    service_client_id: Optional[int] = None
    service_merchant_id: Optional[int] = None
    customer_uid: Optional[int] = None
    return_url: Optional[str] = None
    paymethod_id: Optional[str] = None

    merchant_oauth_mode: MerchantOAuthMode = MerchantOAuthMode.PROD

    # колонка для paytest
    test: Optional[PaymentsTestCase] = None

    email_message_id: Optional[str] = None
    email_context: Optional[dict] = None

    customer_subscription_id: Optional[int] = None
    # колонка для привязки refund-а подписки к конкретной транзакции
    # PK для транзакции состоит из (uid, customer_subscription_id, purchase_token)
    # в этом поле храним purchase_token
    customer_subscription_tx_purchase_token: Optional[str] = None

    offline_abandon_deadline: Optional[datetime] = None

    # miscellaneous attributes, e. g. trust_form_name when create payment via trust client
    data: OrderData = field(default_factory=OrderData)
    exclude_stats: bool = False

    # Generated
    order_hash: Optional[str] = None
    order_url: Optional[str] = None
    payment_hash: Optional[str] = None
    payment_url: Optional[str] = None  # относительная ссылка на заказ/старт оплаты заказа (бэкенд)
    trust_url: Optional[str] = None
    payments_url: Optional[str] = None  # абсолютная ссылка на форму оплаты (на фронте)
    trust_resp_code: Optional[str] = None
    trust_payment_id: Optional[str] = None
    timeline: Optional[List[OrderTimelineEvent]] = None

    # Joined
    merchant: Optional[Merchant] = None
    current_arbitrage: Optional[Arbitrage] = None
    items: Optional[List[Item]] = None
    refunds: Optional[List['Order']] = None
    customer_subscription: Optional[CustomerSubscription] = None
    service_merchant: Optional[ServiceMerchant] = None
    original_order_info: Optional[OriginalOrderInfo] = field(default=None, compare=False)

    # сущность магазина достается в get / find в мапперах, и есть не всегда
    shop: Optional[Shop] = field(default=None, compare=False)
    _watch_pay_status_update: bool = field(default=False, init=False, repr=False, compare=False)

    def __post_init__(self) -> None:
        """
        Трекинг изменения статуса оплаты выключен на время __init__, чтобы инициализация статуса не меняла время
        """
        self._watch_pay_status_update = True

    def __setattr__(self, name: str, value: Any) -> None:
        """
        Обновление времени изменения при смене статуса
        """
        if name == 'pay_status' and self.pay_status != value and self._watch_pay_status_update:
            self.pay_status_updated_at = utcnow()
        super().__setattr__(name, value)

    def add_crypto(self, crypto_prefix: str, crypto: Crypto) -> None:
        if self.order_id is None:
            return

        self.order_hash = order_hash = crypto.encrypt_order(
            uid=self.uid,
            order_id=self.order_id,
        )
        self.order_url = crypto_prefix + order_hash

        self.payment_hash = payment_hash = crypto.encrypt_payment(
            uid=self.uid,
            order_id=self.order_id,
        )
        self.payment_url = crypto_prefix + payment_hash

        front_payment_url = settings.FRONT_PAYMENT_URL.strip('/')
        self.payments_url = f'{front_payment_url}/{order_hash}'

    def get_acquirer(self, default: Optional[AcquirerType]) -> Optional[AcquirerType]:
        return self.acquirer or default

    @property
    def is_test(self) -> bool:
        assert self.shop
        return self.shop.shop_type == ShopType.TEST or self.test is not None

    @property
    def is_already_paid(self) -> bool:
        return PayStatus.is_already_paid(self.pay_status)

    @property
    def pay_method(self) -> Optional[str]:
        if not self.is_already_paid:
            return None
        elif self.paymethod_id == PAYMETHOD_ID_OFFLINE:
            return PAY_METHOD_OFFLINE
        else:
            return PAY_METHOD_YANDEX

    @property
    def price(self) -> Optional[Decimal]:
        if self.items is None:
            return None
        return sum([item.total_price for item in self.items], Decimal(0))

    @property
    def log_price(self) -> float:
        price = self.price
        return 0.0 if price is None else float(round(price, 2))

    @property
    def currency(self) -> Optional[str]:
        if self.items is None:
            return None
        return next(map(lambda item: item.currency, self.items), None)

    @property
    def in_moderation(self) -> bool:
        """При создании модерации для заказа мы переводим заказ в статус (pay_status) IN_MODERATION.
        После обнаружения, что модерация готова, мы обновляем статус заказа.
        Таким образом нахождение заказа в статусе IN_MODERATION должно означать, что модерация для заказа
        была создана, но мы пока не обнаружили либо не получили результат модерации.
        """
        return self.pay_status == PayStatus.IN_MODERATION

    @property
    def is_subscription(self):
        return self.customer_subscription_id is not None

    @property
    def single_item(self) -> Item:
        assert self.items
        items = list(self.items)
        assert len(items) == 1, f'{len(items)} != 1'
        return items[0]

    @property
    def multi_amount_exceed(self) -> bool:
        assert self.kind == OrderKind.MULTI

        if self.data.multi_max_amount is not None and self.data.multi_issued is not None:
            if self.data.multi_issued >= self.data.multi_max_amount:
                return True

        return False

    @property
    def service_data(self) -> Optional[dict]:
        return self.data.service_data

    @property
    def is_create_arbitrage_available(self) -> bool:
        assert self.merchant
        return self.merchant.dialogs_org_id is not None

    @property
    def fast_moderation(self) -> bool:
        return self.data.fast_moderation


@dataclass
class OrdersAdminData:
    orders: List[Order]
    stats: SearchStats
