from __future__ import annotations

from dataclasses import dataclass, field
from datetime import datetime
from typing import List, Optional, Union

from mail.payments.payments.core.entities.enums import AcquirerType, OrderSource
from mail.payments.payments.core.entities.not_fetched import NOT_FETCHED, NotFetchedType
from mail.payments.payments.core.exceptions import OfferSettingSlugInvalidError
from mail.payments.payments.utils.datetime import utcnow


@dataclass
class ServiceMerchant:
    uid: int
    service_id: int
    entity_id: str
    description: str
    service_merchant_id: Optional[int] = None  # autoincremented in DB when created
    enabled: bool = False
    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)
    deleted: bool = False
    revision: Optional[int] = None

    # Joined
    service: Union[NotFetchedType, 'Service'] = field(default=NOT_FETCHED, compare=False, repr=False)


@dataclass
class ServiceClient:
    service_id: int
    tvm_id: int
    api_callback_url: Optional[str] = None
    service_client_id: Optional[int] = None
    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)

    # Joined
    service: Union[NotFetchedType, 'Service'] = field(default=NOT_FETCHED, compare=False, repr=False)


@dataclass
class OfferSettings:  # PAYBACK-622
    pdf_template: Optional[str] = None
    slug: Optional[str] = None
    data_override: dict = field(default_factory=dict)

    def __post_init__(self):
        for acquirer in AcquirerType:
            if self.slug == acquirer.value:
                raise OfferSettingSlugInvalidError


@dataclass
class ServiceOptions:
    offer_settings: OfferSettings = field(default_factory=OfferSettings)  # PAYBACK-622
    service_fee: Optional[int] = None  # PAYBACK-592
    required_acquirer: Optional[AcquirerType] = None  # PAYBACK-625
    require_online: bool = True  # PAYBACK-628
    can_skip_registration: bool = False  # PAYBACK-653
    commission: Optional[int] = None  # PAYBACK-842 в базисных пунктах (1% == 100 б.п.)
    allow_payment_mode_recurrent: bool = False  # PAYBACK-884
    allow_payment_without_3ds: bool = False  # PAYBACK-906

    # Реплики параметров мерчанта, которые проставляются при создании мерчанта сервисом, см. PAYBACK-591, PAYBACK-637
    allowed_order_sources: Optional[List[OrderSource]] = None
    allow_create_service_merchants: bool = True
    hide_commission: bool = False
    icon_url: Optional[str] = None  # PAYBACK-800


@dataclass
class Service:
    name: str
    service_id: Optional[int] = None  # autoincremented in db
    slug: Optional[str] = None
    order_moderation_enabled: bool = True
    hidden: bool = False  # PAYBACK-628
    antifraud: bool = False  # PAYBACK-628
    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)
    options: ServiceOptions = field(default_factory=ServiceOptions)

    # Joined
    service_client: Union[NotFetchedType, ServiceClient] = field(default=NOT_FETCHED, compare=False, repr=False)
    service_merchant: Union[NotFetchedType, ServiceMerchant] = field(default=NOT_FETCHED, compare=False, repr=False)
