from dataclasses import dataclass, field
from datetime import datetime
from decimal import Decimal
from typing import List, Optional

from mail.payments.payments.core.entities.enums import NDS, AcquirerType, MerchantOAuthMode, PeriodUnit
from mail.payments.payments.core.entities.moderation import ModerationData
from mail.payments.payments.utils.helpers import uuid_hex


@dataclass
class SubscriptionPrice:
    price: Decimal
    region_id: int
    currency: str = 'RUB'


@dataclass
class SubscriptionData:
    fast_moderation: bool = False


@dataclass
class Subscription:
    uid: int

    title: str
    fiscal_title: str
    nds: NDS
    period_amount: int
    period_units: PeriodUnit

    prices: List[SubscriptionPrice]

    product_uuid: str = field(default_factory=uuid_hex)
    enabled: bool = True
    service_merchant_id: Optional[int] = None
    service_client_id: Optional[int] = None

    subscription_id: Optional[int] = None
    revision: Optional[int] = None
    created: Optional[datetime] = None
    updated: Optional[datetime] = None
    trial_period_amount: Optional[int] = None
    trial_period_units: Optional[PeriodUnit] = None
    merchant_oauth_mode: MerchantOAuthMode = MerchantOAuthMode.PROD
    acquirer: Optional[AcquirerType] = None  # https://st.yandex-team.ru/PAYBACK-584
    deleted: bool = False
    data: SubscriptionData = field(default_factory=SubscriptionData)

    # Generated
    moderation: Optional[ModerationData] = None
    enabled_customer_subscriptions: Optional[int] = None

    @property
    def product_id(self):
        return f'subs-{self.uid}-{self.subscription_id}-{self.product_uuid}'

    @property
    def period(self) -> str:
        return f'{self.period_amount}{self.period_units.value}'

    @property
    def trial_period(self) -> Optional[str]:
        if self.trial_period_amount and self.trial_period_units:
            return f'{self.trial_period_amount}{self.trial_period_units.value}'
        return None

    @property
    def approx_period_seconds(self) -> int:
        return self.period_amount * self.period_units.approx_seconds

    @property
    def fast_moderation(self):
        return self.data.fast_moderation
