from dataclasses import dataclass, field
from datetime import datetime, timedelta
from math import floor, log
from typing import ClassVar, Dict, List, Optional, Union

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.common import SearchStats
from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.core.entities.not_fetched import NOT_FETCHED, NotFetchedType
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.utils.datetime import utcnow

from .enums import TransactionStatus


@dataclass
class Transaction:
    _CHECK_START_DELAY: ClassVar[Dict[TransactionStatus, int]] = {
        TransactionStatus.ACTIVE: settings.TRANSACTION_CHECK_START_DELAY_ACTIVE,
        TransactionStatus.HELD: settings.TRANSACTION_CHECK_START_DELAY_HELD,
    }
    _CHECK_INITIAL_DELAY: ClassVar[int] = settings.TRANSACTION_CHECK_INITIAL_DELAY
    _CHECK_DELAY_MULTIPLIER: ClassVar[int] = settings.TRANSACTION_CHECK_DELAY_MULTIPLIER
    _CHECK_MAX_DELAY: ClassVar[int] = settings.TRANSACTION_CHECK_MAX_DELAY
    _CHECK_MAX_TRIES: ClassVar[int] = floor(log(_CHECK_MAX_DELAY / _CHECK_INITIAL_DELAY, _CHECK_DELAY_MULTIPLIER))

    uid: int
    order_id: int
    tx_id: Optional[int] = None
    revision: Optional[int] = None
    created: datetime = field(default_factory=utcnow)
    updated: datetime = field(default_factory=utcnow)
    status: TransactionStatus = TransactionStatus.ACTIVE
    trust_purchase_token: Optional[str] = None
    trust_terminal_id: Optional[int] = None
    trust_payment_url: Optional[str] = None
    trust_failed_result: Optional[str] = None
    trust_resp_code: Optional[str] = None
    trust_payment_id: Optional[str] = None
    poll: bool = True
    check_at: datetime = field(default_factory=utcnow)
    check_tries: int = 0

    # Generated
    trust_receipt_download_url: Optional[str] = None
    trust_receipt_view_url: Optional[str] = None

    # Joined
    user_email: Optional[str] = field(default=None, compare=False)
    customer_uid: Optional[int] = field(default=None, compare=False)
    order: Union[NotFetchedType, Order] = field(default=NOT_FETCHED, compare=False, repr=False)

    @property
    def finished(self) -> bool:
        return self.status.is_final

    def reset_check_tries(self) -> None:
        """Called when transaction changes status, resets backoff logic.
        Delays background checks allowing trust to send a callback.
        """
        self.check_at = utcnow() + timedelta(seconds=self._CHECK_START_DELAY.get(self.status, 0))
        self.check_tries = 0

    def increment_check_tries(self) -> None:
        if self.check_tries <= self._CHECK_MAX_TRIES:
            delay = self._CHECK_INITIAL_DELAY * self._CHECK_DELAY_MULTIPLIER ** self.check_tries
        else:
            delay = self._CHECK_MAX_DELAY
        self.check_at = utcnow() + timedelta(seconds=delay)
        self.check_tries += 1

    def set_trust_receipt_urls(self, acquirer: Optional[AcquirerType]) -> None:
        if self.trust_purchase_token and acquirer == AcquirerType.TINKOFF:
            trust_receipt_base_url = settings.TRUST_RECEIPT_BASE_URL.rstrip('/')
            self.trust_receipt_view_url = f'{trust_receipt_base_url}/checks/{self.trust_purchase_token}' \
                                          f'/receipts/{self.trust_purchase_token}/'

            self.trust_receipt_download_url = f'{self.trust_receipt_view_url}?mode=pdf'


@dataclass
class TransactionsAdminData:
    transactions: List[Transaction]
    stats: SearchStats
