from typing import Any, ClassVar, Dict, Optional, Type

from sendr_core.exceptions import BaseCoreError

from mail.payments.payments.conf import settings
from mail.payments.payments.utils.helpers import str_to_underscore


def get_error_message(cls: Type) -> str:
    message = str_to_underscore(cls.__name__).lstrip('_').upper().lstrip('_')
    if message.endswith('_ERROR'):
        message = message[:-len('_ERROR')]
    return message


class CoreDataError(BaseCoreError):
    MESSAGE: ClassVar[Optional[str]] = None

    def __init__(self, message: Optional[str] = None, params: Optional[dict] = None):
        self.message = message or self.MESSAGE
        self.params: dict = params or dict()

    def __init_subclass__(cls):
        if cls.MESSAGE is None:
            cls.MESSAGE = get_error_message(cls)


class CoreFieldError(CoreDataError):
    """
    Example:
    ```
    raise CoreFieldError(
        fields={
            "name": "invalid",
            "persons.ceo.email": Email.default_message
        }
    )
    ```
    In `except` field `fields` will be converted into dict:
    ```
        {
            "persons": {
                "ceo": {
                    "email": [Email.default_message]
                }
            },
            "name": ["invalid"]
        }
    ```
    """

    def __init__(self,
                 message: Optional[str] = None,
                 params: Optional[dict] = None,
                 fields: Optional[Dict[str, str]] = None,
                 ):
        super().__init__(message, params)
        self.fields: Dict[str, Any] = dict()

        if fields:
            for nested_fields, error in fields.items():
                current_leaf = self.fields
                fields_chain = nested_fields.split('.')
                fields_depth = len(fields_chain)

                for current_depth in range(fields_depth - 1):
                    field = fields_chain[current_depth]
                    if field not in current_leaf:
                        current_leaf[field] = dict()
                    current_leaf = current_leaf[field]

                last_field = fields_chain[fields_depth - 1]
                current_leaf[last_field] = list()
                current_leaf = current_leaf[last_field]

                assert isinstance(current_leaf, list)
                current_leaf.append(error)


class CoreInteractionError(BaseCoreError):
    pass


class CoreInteractionTimeoutError(CoreInteractionError):
    pass


class CoreInteractionConnectionTimeoutError(CoreInteractionError):
    pass


class CoreInteractionRequestError(CoreInteractionError):
    pass


class CoreInteractionResponseError(CoreInteractionError):
    pass


class CoreInteractionNotFound(CoreInteractionResponseError):
    pass


class CoreInteractionFatalError(CoreInteractionError):
    pass


class CoreSecurityError(BaseCoreError):
    pass


class CoreAccessDenyError(CoreSecurityError):
    message = 'Access denied'


class CoreActionDenyError(CoreSecurityError):
    message = 'Action access denied'


class CoreUserUIDMismatchError(CoreActionDenyError):
    message = 'USER_UID_MISMATCH'


class ChildMerchantError(CoreActionDenyError):
    message = 'CHILD_MERCHANT_DENY'


class CoreMerchantUserNotAuthorizedError(CoreActionDenyError):
    message = 'MERCHANT_USER_NOT_AUTHORIZED'

    def __init__(self, allowed_roles):
        super().__init__()
        self.params = {'allowed_roles': [role.value for role in allowed_roles]}


class CoreFailError(BaseCoreError):
    pass


class CoreRevisionMismatch(CoreInteractionRequestError):
    pass


class CoreInteractionAlreadyExist(CoreInteractionRequestError):
    pass


class CoreAlreadyExists(CoreDataError):
    def __init__(self, **kwargs):
        super().__init__(params=kwargs)

    def __init_subclass__(cls):
        cls.MESSAGE = get_error_message(cls)


class ServiceMerchantAlreadyExists(CoreAlreadyExists):
    pass


class ServiceMerchantNotEnabledError(CoreActionDenyError):
    message = 'SERVICE_MERCHANT_NOT_ENABLED'


# Not found
class CoreNotFoundError(CoreDataError):
    def __init__(self, **kwargs):
        super().__init__(params=kwargs)

    def __init_subclass__(cls):
        cls.MESSAGE = get_error_message(cls)


class GraphTypeNotFoundError(CoreNotFoundError):
    pass


class DocumentNotFoundError(CoreNotFoundError):
    pass


class ShopNotFoundError(CoreNotFoundError):
    pass


class CategoryNotFoundError(CoreNotFoundError):
    pass


class OrderNotFoundError(CoreNotFoundError):
    pass


class ProductNotFoundError(CoreNotFoundError):
    pass


class ArbitrageNotFoundError(CoreNotFoundError):
    pass


class ManagerActionNotAllowed(CoreActionDenyError):
    pass


class ManagerNotFoundError(CoreNotFoundError):
    pass


class MerchantNotFoundError(CoreNotFoundError):
    pass


class SubscriptionNotFoundError(CoreNotFoundError):
    pass


class CustomerSubscriptionNotFoundError(CoreNotFoundError):
    pass


class CustomerSubscriptionAlreadyCanceledError(CoreDataError):
    pass


class CustomerSubscriptionNotEnabledError(CoreDataError):
    pass


class EnabledCustomerSubscriptionExistsError(CoreDataError):
    pass


class CustomerSubscriptionTransactionNotFoundError(CoreNotFoundError):
    pass


class CustomerSubscriptionTransactionNotClearedError(CoreDataError):
    pass


class CustomerSubscriptionTransactionHasOpenRefundsError(CoreDataError):
    pass


class ReportNotFoundError(CoreNotFoundError):
    pass


class ServiceNotFoundError(CoreNotFoundError):
    MESSAGE = 'SERVICE_NOT_FOUND'


class ServiceMerchantNotFoundError(CoreNotFoundError):
    pass


class TransactionNotFoundError(CoreNotFoundError):
    pass


class UserRoleNotFoundError(CoreNotFoundError):
    pass


# Entities
class UnknownRegionId(CoreDataError):
    pass


class SubscriptionHasNoPriceForRegionId(CoreDataError):
    pass


class DocumentCannotLoadImageError(CoreDataError):
    pass


class DocumentFileSizeLimitExceededError(CoreDataError):
    pass


class DocumentFileTypeNotAllowedError(CoreDataError):
    pass


class DocumentRequestBodyEmptyError(CoreDataError):
    pass


class DocumentTypeNotAllowedError(CoreDataError):
    pass


class DocumentBodyPartError(CoreDataError):
    pass


class CustomerUidInvalid(CoreDataError):
    pass


class SubscriptionRequiresCustomerUid(CoreDataError):
    pass


class MerchantIsAlreadyRegistered(CoreDataError):
    MESSAGE = 'MERCHANT_IS_ALREADY_REGISTERED'


class MerchantNotConnectedToDialogsError(CoreDataError):
    pass


class MerchantCreateDeny(CoreActionDenyError):
    MESSAGE = 'MERCHANT_CREATE_DENY'


class MerchantCannotScheduleModerationForChildError(CoreDataError):
    pass


class MerchantCannotScheduleModerationForPreregisterError(CoreDataError):
    pass


class MerchantCannotUpdateBalanceDataForChildError(CoreDataError):
    pass


class ModerationUnexpectedUnixtimeTypeError(CoreDataError):
    pass


class ModerationAlreadyExistsError(CoreDataError):
    MESSAGE = 'MODERATION_ALREADY_EXISTS'


class OrdersAmountExceed(CoreDataError):
    pass


class OrdersParamsError(CoreDataError):
    pass


class OrderAlreadyHaveTransactions(CoreDataError):
    pass


class OrderHaveUnfinishedTransactions(CoreDataError):
    pass


class OrderCancelledError(CoreDataError):
    pass


class OrderInvalidKind(CoreDataError):
    pass


class OrderAlreadyPaidError(CoreDataError):
    pass


class OrderAlreadyEmailedError(CoreDataError):
    pass


class OrderArchivedError(CoreDataError):
    pass


class OrderCannotBePaidWithoutReturnUrlError(CoreDataError):
    pass


class OrderCannotBePaidWithoutEmailError(CoreDataError):
    pass


class OrderCannotClearOrUnholdRefundError(CoreDataError):
    pass


class OrderHasAutoclearOnError(CoreDataError):
    pass


class OrderHasDuplicateItemEntriesError(CoreDataError):
    pass


class OrderItemNotPresentInOriginalOrderError(CoreDataError):
    pass


class OrderNotAllowedByModerationPolicyError(CoreDataError):
    pass


class OrderAbandonedError(CoreDataError):
    pass


class OrderAbandonDeadlineError(CoreDataError):
    pass


class OrderAbandonProlongationAmountError(CoreDataError):
    pass


class OrderPayStatusMustBeHeldOrInModerationError(CoreDataError):
    pass


class OrderOriginalOrderMustBePayKind(CoreDataError):
    pass


class OrderMustBePaidError(CoreDataError):
    pass


class OrderReceiptTypeMustPrepaid(CoreDataError):
    pass


class OrderInvalidPayMethod(CoreDataError):
    pass


class OrderOriginalOrderMustBePaidError(CoreDataError):
    pass


class OrderAnonymousError(CoreDataError):
    pass


class OrderPriceExceeds100kRUBError(CoreDataError):
    MESSAGE = 'ORDER_PRICE_EXCEEDS_100K_RUB'

    def __init__(self, limit: int):
        super().__init__(params={'max_total_price': limit})


class OrderRefundCannotBePaidError(CoreDataError):
    pass


class TestOrderCannotBeRefundedError(CoreDataError):
    MESSAGE = 'TEST_ORDER_CANNOT_BE_REFUNDED'


class OrderRequestedItemAmountExceedsPaidError(CoreDataError):
    pass


class OrderSourceDeniedError(CoreActionDenyError):
    pass


class BadCommissionError(CoreDataError):
    MESSAGE = 'BAD_COMMISSION_VALUE'


class CommissionDeniedError(CoreDataError):
    MESSAGE = 'COMMISSION_DENIED'


class ItemsInvalidDataError(CoreDataError):
    pass


class ItemsInvalidAmountDataError(CoreDataError):
    pass


class ItemsInvalidTotalPriceDataError(CoreDataError):
    pass


class ItemsInvalidMarkupSumError(CoreDataError):
    pass


class ReportNotUploadedError(CoreDataError):
    MESSAGE = 'REPORT_NOT_UPLOADED'


class UIDNotFound(CoreNotFoundError):
    pass


class LoginNotFound(CoreNotFoundError):
    pass


class PaymethodIdNotFound(CoreNotFoundError):
    pass


class RoleAssignmentNotAllowed(CoreActionDenyError):
    pass


class RoleDeletionNotAllowed(CoreActionDenyError):
    pass


class RecurrentPaymentModeNotAllowed(CoreActionDenyError):
    pass


class PaymentWithout3dsNotAllowed(CoreActionDenyError):
    pass


class OAuthCodeError(CoreDataError):
    MESSAGE = 'CODE_ERROR'


class OAuthAlreadyExistsError(CoreDataError):
    MESSAGE = 'OAUTH_PROD_ALREADY_EXISTS'


class OAuthInvalidGrants(CoreDataError):
    MESSAGE = 'OAUTH_INVALID_GRANTS'


class OAuthAbsentError(CoreDataError):
    MESSAGE = 'OAUTH_ABSENT'

    def __init__(self, *args, **kwargs):
        params = kwargs.pop('params', {})
        params['desc'] = f'You can attach shop on url: {settings.FRONT_OAUTH_SETTINGS_URL}'
        kwargs['params'] = params
        super().__init__(*args, **kwargs)


class ActiveArbitrageAlreadyExistError(CoreDataError):
    pass


class ActiveArbitrageAbsentError(CoreDataError):
    pass


class ArbitrageNotEscalateError(CoreDataError):
    pass


class ShopDefaultMustBePresent(CoreDataError):
    pass


class InnModificationError(CoreDataError):
    MESSAGE = 'INN_MODIFICATION_NOT_ALLOWED'


class InnIsEmptyError(CoreDataError):
    MESSAGE = 'INN_IS_EMPTY'


# Interactions
# developer
class DeveloperKeyAbsentError(CoreAccessDenyError):
    message = 'DEVELOPER_KEY_ABSENT'


class DeveloperKeyAccessDenyError(CoreAccessDenyError):
    message = 'DEVELOPER_KEY_ACCESS_DENY'


# trust
class TrustError(CoreDataError):
    MESSAGE = 'TINKOFF_ERROR'


# so
class SpamError(CoreDataError):
    MESSAGE = 'SPAM'


# tinkoff
class TinkoffMerchantInvalidDataError(CoreDataError):
    MESSAGE = 'TINKOFF_DATA_ERROR'


class TinkoffInvalidSubmerchantIdError(CoreDataError):
    MESSAGE = 'TINKOFF_INVALID_SUBMERCHANT_ID'


class TinkoffMerchantInvalidAddressError(TinkoffMerchantInvalidDataError):
    MESSAGE = 'TINKOFF_ADDRESS_ERROR'


# kassa
class KassaMeError(CoreDataError):
    MESSAGE = 'KASSA_ME_ERROR'


class KeysetInvalidError(CoreDataError):
    """
    Используется для keyset-based pagination при нарушении ограничений на keyset
    """
    MESSAGE = 'KEYSET_INVALID'


class SortByInvalidError(CoreDataError):
    MESSAGE = 'SORT_BY_INVALID_FIELD'


class OfferSettingSlugInvalidError(CoreDataError):
    MESSAGE = 'OFFER_SETTINGS_SLUG_ERROR'


class MerchantExistsPreregisterError(CoreDataError):
    MESSAGE = 'MERCHANT_ALREADY_EXISTS'


class MerchantInactivePreregisterError(CoreDataError):
    MESSAGE = 'MERCHANT_INACTIVE_BY_SPARK'


class UnknownAcquirerType(CoreDataError):
    MESSAGE = 'UNKNOWN_ACQUIRER_TYPE'


class ConflictingAcquirerType(CoreDataError):
    MESSAGE = 'CONFLICTING_ACQUIRER_TYPE'


class MerchantPreregistrationNotFoundError(CoreNotFoundError):
    MESSAGE = 'MERCHANT_PRERGISTRATION_NOT_FOUND'


# Refs
class BankNotFoundError(CoreNotFoundError):
    MESSAGE = 'BANK_NOT_FOUND'


# Functionality validation
class YandexPaySchemaValidationError(CoreDataError):
    MESSAGE = 'YANDEX_PAY_SCHEMA_INVALID'

    def __init__(self, params: Dict[str, Any]):
        self.params = params


class YandexPayPartnerTypeChangedError(CoreDataError):
    MESSAGE = 'YANDEX_PAY_PARTNER_TYPE_CHANGED'


class YandexPayPSPExternalIDChangedError(CoreDataError):
    MESSAGE = 'YANDEX_PAY_PSP_EXTERNAL_ID_CHANGED'


class YandexPayPSPExternalIDEmptyError(CoreDataError):
    message = 'YANDEX_PAY_PSP_EXTERNAL_ID_EMPTY'


class MerchantContactIsEmptyError(CoreDataError):
    message = 'MERCHANT_CONTACT_IS_EMPTY'
