from typing import Optional

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes, serialization
from cryptography.hazmat.primitives.asymmetric import padding


class PartnerCrypto:
    def __init__(self, key_data: bytes, password: Optional[bytes] = None):
        self.private_key = serialization.load_pem_private_key(
            key_data,
            password=password,
            backend=default_backend()
        )
        self.padding = padding.PSS(
            mgf=padding.MGF1(hashes.SHA256()),
            salt_length=padding.PSS.MAX_LENGTH
        )
        self.hash_algo = hashes.SHA256()

    @classmethod
    def from_file(cls, key_path):
        with open(str(key_path), "rb") as key_file:
            return cls(key_data=key_file.read())

    def sign(self, message: bytes) -> bytes:
        return self.private_key.sign(message, self.padding, self.hash_algo)
