from typing import List

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.arbitrage import Arbitrage
from mail.payments.payments.core.entities.enums import NDS, PAY_METHOD_OFFLINE
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.floyd.entities import Message, MessageActor
from mail.payments.payments.utils.helpers import without_none


class ArbiterClient(AbstractInteractionClient):
    """
    Клиент для создания арбитража для заказа.
    Спецификация АПИ: https://arbiter.tst.vs.market.yandex.net/swagger-ui.html.
    """

    SERVICE = 'arbiter'
    BASE_URL = settings.ARBITER_URL.rstrip('/')
    TVM_ID = settings.TVM_ARBITER_CLIENT_ID

    async def create_arbitrage(
        self,
        merchant: Merchant,
        order: Order,
        arbitrage: Arbitrage,
        messages: List[Message],
    ) -> int:
        assert (
            order.closed is not None
            and order.price is not None
            and order.customer_uid is not None
        )

        result = await self.post(
            'create_arbitrage',
            self.endpoint_url('api/v1/service/conversation/add'),
            json={
                "description": "Пользователь хочет получить деньги назад.",
                "merchant": without_none({
                    "idInService": merchant.merchant_id,
                    "inn": merchant.organization.inn,
                    "name": merchant.organization.full_name,
                    "ogrn": merchant.organization.ogrn,
                }),
                "messages": [
                    {
                        'id': message.message_id,
                        'creationTime': message.creation_time.isoformat(),
                        'attachments': message.attachments,
                        'sender': 'MERCHANT' if message.sender == MessageActor.OPERATOR else 'USER',
                        'recipient': 'MERCHANT' if message.recipient == MessageActor.OPERATOR else 'USER',
                        'text': message.text,
                    }
                    for message in messages
                ],
                "serviceType": "SUPERAPP",
                "subject": {
                    "currency": order.currency,
                    "description": order.description,
                    "idInService": f"{order.uid}.{order.order_id}",
                    "orderInfo": {
                        "history": [
                            {
                                "status": "DELIVERED",
                                "time": order.closed.isoformat()
                            }
                        ],
                        "items": [
                            {
                                "currency": item.currency,
                                "idInService": f'{item.uid}.{item.order_id}.{item.product_id}',
                                "name": item.product.name,  # type: ignore
                                "price": int(item.product.price * 100),  # type: ignore
                                "quantity": float(item.amount),
                                "totalCost": int(item.price * 100),  # type: ignore
                                "vat": NDS.to_arbitrage(item.nds)  # type: ignore
                            } for item in order.items  # type: ignore
                        ]
                    },
                    "paymentType": "CASH" if order.pay_method == PAY_METHOD_OFFLINE else "CREDIT_CARD",
                    "totalCost": int(order.price * 100),
                    "type": "ORDER"
                },
                "uid": order.customer_uid,
                "notificationChannels": [
                    {
                        'businesschatParams': {
                            'recipientId': arbitrage.chat_id,
                            'chatId': settings.ARBITER_CHAT_ID
                        },
                        'conversationSide': 'USER',
                        'type': 'BUSINESSCHAT'
                    },
                    {
                        'businesschatParams': {
                            'recipientId': arbitrage.arbiter_chat_id,
                            'chatId': settings.ARBITER_CHAT_ID
                        },
                        'conversationSide': 'MERCHANT',
                        'type': 'BUSINESSCHAT'
                    }
                ]
            }
        )
        return result['id']

    async def execute_verdict(self, conversation_id: int) -> None:
        await self.post(
            'execute_verdict',
            self.endpoint_url('api/v1/service/verdict/execute'),
            params={'conversationId': conversation_id}
        )
