from typing import Any, Optional, Union
from xml.etree import ElementTree
from xmlrpc import client as xmlrpc

from aiohttp import ClientResponse

from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.utils.stats import interaction_method_response_status, interaction_response_status

from .exceptions import BaseBalanceError


class BaseBalanceClient(AbstractInteractionClient):
    """
    Base client for Balance xmlrpc API
    https://wiki.yandex-team.ru/balance/xmlrpc/

    Note: some methods throw fault on error, and some return error code.
    """

    def _handle_error_codes(self, *codes: Union[str, int, None], message: Optional[str] = None) -> None:
        if codes == (0,):
            return
        error = BaseBalanceError.get_error_by_codes(codes)
        if error is not None:
            raise error(message=message)
        with self.logger:
            self.logger.context_push(codes=codes)
            self.logger.error('Unknown balance error')
        raise BaseBalanceError(message=f'Unknown balance error. Codes: {codes}')

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> Any:
        data = await response.read()
        try:
            data = xmlrpc.loads(data)
        except xmlrpc.Fault as fault:
            interaction_method_response_status.labels('balance', interaction_method, 'faults').inc()
            interaction_response_status.labels('balance', 'faults').inc()
            try:
                fault_xml = ElementTree.fromstring(fault.faultString)
                codes = [code.text for code in fault_xml.findall('code')]
                error_message = fault_xml.findtext('msg')
            except ElementTree.ParseError:
                codes = []
                error_message = None
                self.logger.exception('Unable to parse balance fault')
            self._handle_error_codes(*codes, message=error_message)
        return data

    async def _make_request(self, *args, **kwargs):
        kwargs['data'] = xmlrpc.dumps(kwargs['data'], kwargs.pop('method_name'))
        return await super()._make_request(*args, **kwargs)

    async def request(self, interaction_method, **kwargs):
        return (await self.post(interaction_method, self.BASE_URL, **kwargs))[0]
