from typing import List, Optional

from mail.payments.payments.interactions.balance.base import BaseBalanceClient


class BalanceContractClient(BaseBalanceClient):
    async def get_client_contracts(self,
                                   client_id: str,
                                   person_id: Optional[str] = None,
                                   is_active: Optional[bool] = None,
                                   external_id: Optional[str] = None) -> List[dict]:
        """
        Get list of all client contracts.
        Args:
            client_id: Balance ClientID
            person_id: Balance PersonID to filter contracts on Balance side
            is_active: filter returned contracts by is_active flag
            external_id: Balance ExternalId to filter contracts on Balance side
        Returns:
            List of contracts. Each contract looks like:
            {
                'CURRENCY': 'RUR',
                'OFFER_ACCEPTED': 1,
                'IS_SUSPENDED': 0,
                'MANAGER_CODE': 12345,
                'IS_ACTIVE': 1,
                'IS_SIGNED': 1,
                'CONTRACT_TYPE': 9,
                'PERSON_ID': 9268246,
                'IS_FAXED': 0,
                'SERVICES': [625],
                'PAYMENT_TYPE': 3,
                'PARTNER_COMMISSION_PCT2': '2.15',
                'IS_CANCELLED': 0,
                'IS_DEACTIVATED': 0,
                'DT': <DateTime '20190822T00:00:00' at 0x7fb6511b9f98>,
                'EXTERNAL_ID': '406024/19',
                'ID': 1027802
            }
        """
        data = {'ClientID': client_id}
        if person_id is not None:
            data['PersonID'] = person_id
        if external_id is not None:
            data['ExternalID'] = external_id

        result, = await self.request(
            'get_client_contracts',
            method_name='GetClientContracts',
            data=(data,),
        )

        if is_active is not None:
            result = [item for item in result if item['IS_ACTIVE'] == int(is_active)]

        return result
