from datetime import datetime
from typing import Optional, Tuple

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import AcquirerType

from .base import BaseBalanceClient


class BalanceOfferClient(BaseBalanceClient):
    async def create_offer(self,
                           uid: int,
                           acquirer: AcquirerType,
                           client_id: str,
                           person_id: str,
                           extra_params: Optional[dict] = None) -> Tuple[int, str]:
        """
        Creates offer in balance.
        Returns:
            ID, EXTERNAL_ID pair for created Offer
        """
        assert acquirer.value in settings.BALANCE_SERVICE_ID, f'Unknown service_id for {acquirer}'

        offer_data = {
            'client_id': client_id,
            'person_id': person_id,
            'services': (settings.BALANCE_SERVICE_ID[acquirer.value],),
            'payment_type': settings.BALANCE_PAYMENT_TYPE,
            'partner_credit': settings.BALANCE_PARTNER_CREDIT,
            'firm_id': settings.BALANCE_FIRM_ID,
            'currency': settings.BALANCE_CURRENCY,
            'payment_term': settings.BALANCE_PAYMENT_TERM,
            'offer_confirmation_type': settings.BALANCE_OFFER_CONFIRMATION_TYPE,
            'partner_commission_pct2': settings.BALANCE_COMISSION,
            'manager_code': settings.BALANCE_MANAGER_CODE,
            **(extra_params or {})
        }
        result = await self.request(
            'create_offer',
            method_name='CreateOffer',
            data=(str(uid), offer_data),
        )
        contract_data, = result
        return contract_data['ID'], str(contract_data['EXTERNAL_ID'])

    async def create_collateral(self, uid: int, contract_id: str) -> int:
        """
        Creates collateral for existing contract. Returns it's id.
        """
        collateral_data = {'END_DT': datetime.now()}
        result = await self.request(
            'create_collateral',
            method_name='CreateCollateral',
            data=(str(uid), contract_id, settings.BALANCE_COLLATERAL_TYPE, collateral_data),
        )
        contract_data, = result
        return contract_data['CONTRACT_ID']
