from typing import Any, Dict, Iterable, Optional, Union

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.userinfo import UserInfo
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.exceptions import BlackBoxDefaultEmailNotFoundError, BlackBoxUserNotFoundError
from mail.payments.payments.utils.helpers import without_none

IntStr = Union[str, int]
UidType = Union[Iterable[IntStr], IntStr]


class BlackBoxClient(AbstractInteractionClient):
    SERVICE = 'BlackBox'
    BASE_URL = settings.BLACKBOX_API_URL
    TVM_ID = settings.TVM_BLACKBOX_CLIENT_ID

    def _uid_param(self, uids: Optional[UidType] = None) -> Optional[str]:
        uid = None
        if isinstance(uids, (int, str)):
            uid = str(uids)
        elif uids is not None and bool(uids):
            uid = ','.join(str(uid) for uid in uids)
        return uid

    async def _userinfo(
        self,
        uids: Optional[UidType] = None,
        login: Optional[str] = None,
        userip: str = '127.0.0.1',
        sid: Optional[str] = None,
        get_default_emails: Optional[bool] = False,
    ) -> Dict[str, Any]:
        """
        https://docs.yandex-team.ru/blackbox/methods/userinfo
        """
        assert uids is not None or login is not None
        emails = 'getdefault' if get_default_emails else None
        params = without_none({
            'method': 'userinfo',
            'format': 'json',
            'userip': userip,
            'sid': sid,
            'uid': self._uid_param(uids=uids),
            'login': login,
            'emails': emails,
        })
        data: Dict[str, Any] = await self.get('userinfo', self.BASE_URL, params=params)
        if not data.get('users') or not data['users'][0].get('id'):
            raise BlackBoxUserNotFoundError(service=self.SERVICE, method='get_userinfo')
        return data['users'][0]

    def _uid(self, user: Dict[str, Any]) -> int:
        """Shortcut to request UID by passport login"""
        return int(user['id'])

    def _default_email(self, user: Dict[str, Any]) -> str:
        emails = user.get('address-list') or []
        for email in emails:
            if email['default']:
                return email['address']
        raise BlackBoxDefaultEmailNotFoundError(service=self.SERVICE, method='get_default_email')

    async def userinfo(
        self,
        uids: Optional[UidType] = None,
        login: Optional[str] = None,
        userip: str = '127.0.0.1',
        sid: Optional[str] = None,
        get_default_emails: Optional[bool] = False,
    ) -> UserInfo:
        user = await self._userinfo(uids, login, userip, sid, get_default_emails)
        uid = self._uid(user)
        try:
            default_email = self._default_email(user) if get_default_emails else None
        except BlackBoxDefaultEmailNotFoundError:
            raise BlackBoxDefaultEmailNotFoundError(service=self.SERVICE, method='get_default_email')
        return UserInfo(uid=uid, default_email=default_email)


class BlackBoxCorpClient(BlackBoxClient):
    SERVICE = 'BlackBoxCorp'
    BASE_URL = settings.BLACKBOX_CORP_API_URL
    TVM_ID = settings.TVM_BLACKBOX_CORP_CLIENT_ID
