from base64 import b64encode
from typing import Any, Callable, ClassVar, Dict

import aiohttp
import jwt
import ujson
from aiohttp import ClientResponse

from sendr_utils import json_value

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.zora import AbstractZoraClient


class CallbackClient(AbstractZoraClient):
    SERVICE = 'callback'
    REQUEST_RETRY_TIMEOUTS = ()

    ZORA_VERIFY_CERT = False

    TOTAL_TIMEOUT: ClassVar[int] = settings.MERCHANT_CALLBACK_HTTP_TOTAL_TIMEOUT
    CONNECTION_TIMEOUT: ClassVar[int] = settings.MERCHANT_CALLBACK_HTTP_CONNECTION_TIMEOUT

    def __init__(self, *args: Any, **kwargs: Any) -> None:
        super().__init__(*args, **kwargs)
        self._timeout = aiohttp.ClientTimeout(
            total=self.TOTAL_TIMEOUT,
            sock_connect=self.CONNECTION_TIMEOUT,
        )

    async def _process_response(self,
                                response: ClientResponse,
                                interaction_method: str) -> ClientResponse:
        if response.status >= 400:
            await self._handle_response_error(response)
        return response

    def _get_session_kwargs(self) -> dict:
        return {
            **super()._get_session_kwargs(),
            'timeout': self._timeout,
        }

    def sign_message(self, message: str, signer: Callable[[bytes], bytes]) -> str:
        return b64encode(signer(message.encode('utf-8'))).decode('utf-8')

    async def _post_message(self, url: str, message: str) -> ClientResponse:
        return await self.post(
            'post_message',
            url,
            json={
                'message': message,
            },
        )

    async def _post_signed_message(self, url: str, message: str, sign: str) -> ClientResponse:
        return await self.post(
            'post_signed_message',
            url,
            json={
                'message': message,
                'sign': sign,
            },
        )

    async def post_signed_message(self, url: str, message: Dict[str, Any],
                                  signer: Callable[[bytes], bytes]) -> ClientResponse:
        message_dump: str = ujson.dumps(json_value(message))
        sign: str = self.sign_message(message_dump, signer)
        return await self._post_signed_message(url=url, message=message_dump, sign=sign)

    async def post_jwt_message(self, url: str, message: Dict[str, Any], secret: str) -> ClientResponse:
        message_dump: str = str(jwt.encode(json_value(message), secret, algorithm='HS256'))
        return await self._post_message(url=url, message=message_dump)
