from typing import ClassVar

import ujson
from aiohttp import ClientResponse

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.base import AbstractInteractionClient

from .exceptions import DeveloperKeyAccessDeny, DeveloperServiceNotFound


class DeveloperClient(AbstractInteractionClient):
    """
    Client for developer.tech.yandex.ru API.

    For some reason API responds with mimetype text/html though respnose always contains json.
    """
    SERVICE = 'developer'
    BASE_URL = settings.DEVELOPER_URL.rstrip('/')

    _service_token: ClassVar[str] = settings.DEVELOPER_SERVICE_TOKEN

    async def _handle_response_error(self, response):
        if response.status == 404:
            data = ujson.loads(await response.read())
            error = data.get('error')
            if error == 'Key not found':
                raise DeveloperKeyAccessDeny
            elif error == 'Service not found':
                self.logger.error(
                    'Developer client service not found. Make sure that service token is valid'
                )
                raise DeveloperServiceNotFound

        if response.status == 403:
            raise DeveloperKeyAccessDeny
        await super()._handle_response_error(response)

    async def _process_response(self,
                                response: ClientResponse,
                                interaction_method: str) -> ClientResponse:
        if response.status >= 400:
            await self._handle_response_error(response)
        return ujson.loads(await response.read())

    async def check_key(self, key: str, user_ip: str) -> int:
        """
        https://doc.yandex-team.ru/developer/start-guide/api/check_key.html
        """
        url = self.endpoint_url('api/check_key')

        response = await self.get(
            'check_key',
            url,
            params={
                'service_token': self._service_token,
                'key': key,
                'user_ip': user_ip,
            }
        )

        if settings.DEVELOPER_OVERWRITE_UID:
            return settings.DEVELOPER_OVERWRITE_UID
        return response['key_info']['user']['uid']
