from dataclasses import asdict
from datetime import datetime, timezone
from json.decoder import JSONDecodeError
from typing import Any, List, Optional, Tuple

from aiohttp import ContentTypeError

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.floyd.entities import Message, MessageActor
from mail.payments.payments.utils.helpers import uuid_hex


class FloydClient(AbstractInteractionClient):
    # https://wiki.yandex-team.ru/chats/consult/
    SERVICE = 'floyd'
    BASE_URL = settings.FLOYD_URL.rstrip('/')
    TVM_ID = settings.TVM_FLOYD_CLIENT_ID

    def endpoint_url(self, org_id: str, base_url_override: Optional[str] = None) -> str:
        return super().endpoint_url(f'{org_id}/jsonrpc/', base_url_override=base_url_override)

    async def _process_response(self, response: Any, interaction_method: str) -> dict:
        try:
            data = await response.json()
        except (JSONDecodeError, ContentTypeError):
            await self._handle_response_error(response)

        if response.status >= 400 or 'error' in data:
            await self._handle_response_error(response)

        return data

    async def _request(self, interaction_method, method, org_id, **kwargs):
        json = kwargs.pop('json', {})
        json.update({'jsonrpc': '2.0', 'id': uuid_hex()})
        kwargs['json'] = json

        return await super()._request(interaction_method, 'POST', self.endpoint_url(org_id), **kwargs)

    async def create_consultation(self, org_id: str, description: str, order: Order) -> dict:
        assert order.customer_uid is not None
        result = await self.post('create_consultation', org_id, json={
            'method': 'createConsultation',
            'params': {
                'scenario_id': settings.FLOYD_SCENARIO_ID,
                'description': description,
                'clients': [
                    {'user_id': {'puid': order.customer_uid}, 'role_id': 'client'},
                ],
                'params': {
                    'org_id': org_id,
                    'description': description,
                    'arbitrage_iframe_link': settings.FLOYD_ARBITRAGE_IFRAME_LINK.format(**asdict(order)),
                }
            }
        })
        return result['result']

    async def update_chatbar(self, org_id: str, consultation_id: str, role_id: str, chatbar: dict) -> dict:
        return await self.post('update_chatbar', org_id, json={
            'method': 'updateChatBar',
            'params': {
                'consultation_id': consultation_id,
                'role_id': role_id,
                'chatbar': chatbar
            }
        })

    async def upload_scenario(self, content: str) -> dict:
        return await self.post('upload_scenario', settings.FLOYD_ORG_ID, json={
            'method': 'uploadScenario',
            'params': {
                'scenario_id': settings.FLOYD_SCENARIO_ID,
                'content': content
            }
        })

    async def create_organizations_chat(self,
                                        operator_organization_id: str,
                                        client_organization_id: str) -> Tuple[str, str]:
        result = await self.post('create_organizations_chat', settings.FLOYD_ORG_ID, json={
            'method': 'createOrganizationsChat',
            'params': {
                'operator_organization_id': operator_organization_id,
                'client_organization_id': client_organization_id
            }
        })

        return result['result']['clients']['chat_id'], result['result']['operators']['chat_id']

    async def get_consultation_history(self, org_id: str, consultation_id: str) -> List[Message]:
        result = await self.post('get_consultation_history', org_id, json={
            'method': 'getConsultationHistory',
            'params': {
                'consultation_id': consultation_id
            }
        })

        messages = []
        for item in result['result']['client_messages'] + result['result']['operator_messages']:
            attachments: List[dict] = []

            if item['body'].get('image'):
                attachments.append({"url": item['body']['image']['url']})
            if item['body'].get('file'):
                attachments.append({"url": item['body']['file']['url']})

            message = Message(
                message_id=int(item['id']),
                text=item['body'].get('text') or '',
                sender=MessageActor.CLIENT if item['sender']['is_client'] else MessageActor.OPERATOR,
                recipient=MessageActor.OPERATOR if item['sender']['is_client'] else MessageActor.CLIENT,
                creation_time=datetime.fromtimestamp(item['timestamp'] / 1000).astimezone(timezone.utc),
                attachments=attachments
            )
            messages.append(message)

        return sorted(messages, key=lambda x: x.creation_time)

    async def custom_action(self, org_id: str, consultation_id: str, action: str,
                            params: Optional[dict] = None) -> dict:
        result = await self.post('custom_action', org_id, json={
            'method': 'customAction',
            'params': {
                'consultation_id': consultation_id,
                'action': action,
                'params': {
                    'params': params or {}
                }
            }
        })
        return result['result']

    async def send_div_card_message(self, org_id: str, consultation_id: str, card: dict,
                                    message: Optional[str] = None, to_role_ids: Optional[List[str]] = None) -> dict:
        result = await self.post('send_div_card_message', org_id, json={
            'method': 'sendDivCardMessage',
            'params': {
                'consultation_id': consultation_id,
                'card': card,
                'message': message or '',
                'to_role_ids': to_role_ids
            }
        })

        return result['result']
