from typing import List

from aiohttp import ContentTypeError

from sendr_interactions import exceptions as interaction_errors

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.base import AbstractInteractionClient


class GeobaseClient(AbstractInteractionClient):
    # https://doc.yandex-team.ru/lib/libgeobase5/concepts/region-types.html
    REGION_TYPE_COUNTRY = 3
    SERVICE = 'geobase'
    BASE_URL = settings.GEOBASE_URL.rstrip('/')

    async def _handle_response_error(self, response):
        try:
            error = (await response.json())['error']
        except ContentTypeError:
            error = 'Invalid response from geobase'

        raise interaction_errors.InteractionResponseError(
            status_code=response.status,
            response_status=None,
            params={'error': error},
            service=self.SERVICE,
            method=response.method,
        )

    async def get_region_id_by_ip(self, ip: str) -> int:
        result = await self.get(interaction_method='get_region_id_by_ip',
                                url=self.endpoint_url('v1/get_traits_by_ip'),
                                params={'ip': ip})

        return result['region_id']

    async def get_region(self, region_id: int) -> dict:
        return await self.get(interaction_method='get_region',
                              url=self.endpoint_url('v1/region_by_id'),
                              params={'id': region_id})

    async def is_in(self, region_id: int, in_region_id: int) -> bool:
        """
        :param region_id: какой регион входит, например, Москва — 213
        :param in_region_id: в регион, например, Россия — 225
        :return: true (Москва в России)
        """
        return await self.get(interaction_method='in',
                              url=self.endpoint_url('v1/in'),
                              params={'id': region_id, 'pid': in_region_id})

    async def get_parents(self, region_id: int) -> List[int]:
        return await self.get(interaction_method='parents',
                              url=self.endpoint_url('v1/parents'),
                              params={'id': region_id})
