import uuid
from typing import AsyncIterable, ClassVar, Optional, Tuple

import backoff
import xmltodict
from aiohttp import ClientResponse

from sendr_interactions import exceptions as interaction_errors

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.base import AbstractInteractionClient


class MDSClient(AbstractInteractionClient):
    SERVICE = 'mds'
    TVM_ID = settings.TVM_MDS_CLIENT_ID

    READ_BASE_URL: ClassVar[str] = settings.MDS_READ_URL
    WRITE_BASE_URL: ClassVar[str] = settings.MDS_WRITE_URL
    NAMESPACE: ClassVar[str]
    EXPIRE: ClassVar[Optional[str]] = None

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> ClientResponse:
        if response.status >= 400:
            await self._handle_response_error(response)
        return response

    def read_url(self, relative_url: str) -> str:
        return self.endpoint_url(relative_url, base_url_override=self.READ_BASE_URL)

    def write_url(self, relative_url: str) -> str:
        return self.endpoint_url(relative_url, base_url_override=self.WRITE_BASE_URL)

    def download_url(self, filepath: str) -> str:
        return self.read_url(f'get-{self.NAMESPACE}/{filepath}')

    async def download(self, filepath: str, close: bool = False) -> Tuple[Optional[str], AsyncIterable[bytes]]:
        """
        Can be used with web.StreamResponse to stream response inside handler.
        To do this `close` must be set to True so that session is closed once data is read.
        """
        url = self.download_url(filepath)
        response = await self.get('download', url, response_log_body=False)

        async def _iter_data():
            async for data, _ in response.content.iter_chunks():
                yield data
            if close:
                await self.close()

        return response.headers.get('Content-Type'), _iter_data()

    @backoff.on_exception(
        backoff.constant,
        interaction_errors.InteractionResponseError,
        giveup=lambda e: e.status_code != 403,
        max_tries=settings.MDS_UPLOAD_RETRIES,
        interval=0,
    )
    async def upload(self, filename: str, data: bytes) -> str:
        url = self.write_url(f'upload-{self.NAMESPACE}/{filename}.{uuid.uuid4().hex}')
        params = None if self.EXPIRE is None else {'expire': self.EXPIRE}
        response = await self.post(
            'upload',
            url,
            data=data,
            params=params,
        )
        response_data = xmltodict.parse(await response.read(), dict_constructor=dict)
        return response_data['post']['@key']

    async def remove(self, filepath: str) -> None:
        url = self.write_url(f'delete-{self.NAMESPACE}/{filepath}')
        await self.get('remove', url)


class PaymentsMDSClient(MDSClient):
    service = 'payments_mds'
    NAMESPACE = settings.MDS_PAYMENTS_NAMESPACE
    EXPIRE = None


class ModerationMDSClient(MDSClient):
    service = 'moderation_mds'
    NAMESPACE = settings.MDS_MODERATION_NAMESPACE
    EXPIRE = settings.MDS_MODERATION_EXPIRE
