from typing import Any, Dict

from aiohttp import BasicAuth, ClientResponse, ContentTypeError

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.merchant_oauth import OAuthToken
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.exceptions import OAuthClientError, OAuthInvalidGrantsClientError


class OAuthClient(AbstractInteractionClient):
    # https://yandex.ru/dev/oauth/doc/dg/concepts/about-docpage/
    SERVICE = 'oauth'
    BASE_URL = settings.OAUTH_URL.rstrip('/')

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> dict:
        try:
            data = await response.json()
        except ContentTypeError:
            content = await response.text()
            with self.logger:
                self.logger.context_push(content=content)
                self.logger.error('OAuth decode failed.')
            raise OAuthClientError(method=response.method, message='OAuth response decode fail.')

        if 'error' in data or response.status >= 400:
            message = 'OAuth error',
            params = {'status': response.status, **data}

            if data['error'] == 'invalid_grant':
                raise OAuthInvalidGrantsClientError(method=response.method, message=message, params=params)
            raise OAuthClientError(method=response.method, message=message, params=params)

        return data

    def _get_session_kwargs(self) -> Dict[str, Any]:
        login, password = settings.OAUTH_APP_ID, settings.OAUTH_APP_PASSWORD
        assert all((login, password)), 'OAUTH_APP_ID and OAUTH_APP_PASSWORD must be present'

        kwargs = super()._get_session_kwargs()
        kwargs['auth'] = BasicAuth(login=login, password=password)
        return kwargs

    # https://kassa.yandex.ru/developers/partners-api/oauth#request-token
    # Начать флоу: https://oauth.yandex.ru/authorize?response_type=code&client_id=OAUTH_APP_ID
    async def _get_token(self, grant_type: str, grant_type_param: str, token: str) -> OAuthToken:
        data = {'grant_type': grant_type, grant_type_param: token}
        res = await self.post('_get_token', self.endpoint_url('token'), data=data)

        return OAuthToken(
            token_type=res['token_type'],
            access_token=res['access_token'],
            refresh_token=res['refresh_token'],
            expires_in=res['expires_in']
        )

    async def get_token(self, code: str) -> OAuthToken:
        return await self._get_token('authorization_code', 'code', code)

    async def refresh_token(self, refresh_token: str) -> OAuthToken:
        return await self._get_token('refresh_token', 'refresh_token', refresh_token)
