from typing import Any, Dict, List

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.order import Order, RefundStatus
from mail.payments.payments.core.entities.service import ServiceMerchant
from mail.payments.payments.core.entities.transaction import Transaction, TransactionStatus
from mail.payments.payments.interactions.base import AbstractInteractionClient


class OhioClient(AbstractInteractionClient):
    SERVICE = 'ohio'
    BASE_URL = settings.OHIO_URL.rstrip('/')
    TVM_ID = settings.TVM_OHIO_CLIENT_ID

    @staticmethod
    def _dump_order_status(transaction: Transaction, refunds: List[Order]) -> str:
        if any((refund.refund_status == RefundStatus.COMPLETED for refund in refunds)):
            return 'refunded'
        assert transaction.status.was_held, 'Order transaction must pass held status.'
        if transaction.status in (TransactionStatus.HELD, TransactionStatus.CLEARED):
            return 'paid'
        elif transaction.status in (TransactionStatus.CANCELLED,):
            return 'cancelled'
        else:
            assert False, f'Unexpected transaction status: {transaction.status.value}.'

    @staticmethod
    def _dump_refund_status(refund: Order) -> str:
        assert refund.refund_status is not None, 'Refund is missing refund_status.'
        return refund.refund_status.value

    @staticmethod
    def _dump_item(item: Item) -> Dict[str, Any]:
        assert item.product is not None, 'Item is missing product.'
        data = {
            'image_path': None,
            'image_url': None,
            'name': item.product.name,
            'nds': item.product.nds.value,
            'amount': str(item.amount),
            'price': str(item.product.price),
            'currency': item.product.currency,
        }

        image = item.image
        if image is not None:
            data['image_url'] = image.url
            if image.stored:
                data['image_path'] = image.stored.path
        return data

    @classmethod
    def _dump_refund(cls, refund: Order) -> Dict[str, Any]:
        total = refund.price
        assert total is not None, 'Refund total is None.'
        currency = refund.currency
        assert currency is not None, 'Refund currency is None.'
        assert refund.trust_refund_id is not None, 'Refund is missing trust_refund_id.'
        assert refund.items is not None, 'Refund is missing items.'
        return {
            'trust_refund_id': refund.trust_refund_id,
            'refund_status': cls._dump_refund_status(refund),
            'total': str(total),
            'currency': currency,
            'items': [cls._dump_item(item) for item in refund.items]
        }

    @classmethod
    def _dump_order_data(cls, order: Order, refunds: List[Order]) -> Dict[str, Any]:
        total = order.price
        assert total is not None, 'Order total is None.'
        currency = order.currency
        assert currency is not None, 'Order currency is None.'
        assert order.items is not None, 'Order is missing items.'
        return {
            'total': str(total),
            'currency': currency,
            'description': order.description or '',
            'items': [cls._dump_item(item) for item in order.items],
            'refunds': [cls._dump_refund(refund) for refund in refunds],
        }

    async def post_order(self,
                         order: Order,
                         transaction: Transaction,
                         refunds: List[Order],
                         service_merchant: ServiceMerchant,
                         revision: int,
                         ) -> None:
        """
        Sends order to Ohio. Order and refunds must have items and products loaded. Transaction must be either finished
        successfully or cancelled. All refunds must be created in Trust and have trust_refund_id.
        """
        assert order.customer_uid is not None, 'Order is missing customer_uid.'
        assert transaction.trust_purchase_token is not None, 'Transaction is missing trust_purchase_token.'

        await self.post(
            interaction_method='v1_internal_customer_orders',
            url=self.endpoint_url(f'v1/internal/customer/{order.customer_uid}/orders'),
            json={
                'payments_service_id': service_merchant.service_id,
                'subservice_id': service_merchant.entity_id,
                'merchant_uid': service_merchant.uid,
                'service_merchant_id': service_merchant.service_merchant_id,
                'trust_purchase_token': transaction.trust_purchase_token,
                'created': transaction.created.isoformat(),
                'revision': revision,
                'payments_order_id': order.order_id,
                'order_data': self._dump_order_data(order, refunds),
                'status': self._dump_order_status(transaction, refunds),
                'service_data': order.data.service_data or {},
            }
        )
