import re
from typing import Any, Callable, Dict, List, Optional

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.search_wizard.entities import Fio


class SearchWizardClient(AbstractInteractionClient):
    """ Поисковый wizard: https://wiki.yandex-team.ru/poiskovajaplatforma/lingvistika/wizard/wizardasservis/
    Используется только функциональность разделения ФИО на части """

    SERVICE = 'search_wizard'
    BASE_URL = settings.SEARCH_WIZARD_URL

    def _get_session_kwargs(self) -> dict:
        return {
            **super()._get_session_kwargs(),
            **self._get_timeout_kwargs(settings.SEARCH_WIZARD_TIMEOUT),
        }

    def _parse_json_response(self, json: Dict[str, Any]) -> Optional[Fio]:
        if 'Fio' not in json:
            return None

        fio_items = json['Fio']
        if not fio_items:
            return None

        name_parts = fio_items[0]

        if 'FirstName' not in name_parts or 'LastName' not in name_parts:
            self.logger.warning('First or last name is missing')
            return None

        first_name = name_parts['FirstName']
        middle_name = None
        last_name = name_parts['LastName']
        if 'Patronymic' in name_parts:
            middle_name = name_parts['Patronymic']
        return Fio(first_name, middle_name, last_name)

    async def split_fio(self, fio: str) -> Optional[Fio]:
        assert fio

        params = {
            'action': 'markup',
            'markup=layers': 'Fio',
            'text': fio,
            'wizclient': 'payments',
        }
        response = await self.get('search_wizard', self.BASE_URL, params=params)
        wizard_fio = self._parse_json_response(response)

        if not wizard_fio:
            return None
        return SearchWizardClient._match_result_with_input(fio, wizard_fio)

    @staticmethod
    def _match_result_with_input(input_fio: str, wizard_fio: Fio) -> Optional[Fio]:
        """ Полученный ответ Визарда сопоставляется с исходной строкой. Каждая составляющая имени продлевается
        до начала следующей. Позволяет ничего не упустить из иностранных имён из нескольких слов и корректно
        обработать двойные фамилии через дефис (Визард возвращает только 1-ю часть) """

        assert input_fio
        assert wizard_fio

        input_fio = input_fio.replace('.', ' ')
        input_tokens = [(m[0]) for m in re.finditer(r'\S+', input_fio)]
        input_tokens_lower = [x.lower() for x in input_tokens]

        wizard_tokens_with_title = [
            (wizard_fio.first_name.lower(), 'first_name'),
            (wizard_fio.last_name.lower(), 'last_name')
        ]
        if wizard_fio.middle_name:
            wizard_tokens_with_title.append((wizard_fio.middle_name.lower(), 'middle_name'))

        # process name parts in descending order by length to avoid collisions because of same prefix
        wizard_tokens_with_title.sort(key=lambda field: len(field[0]), reverse=True)

        seen_indexes = set()

        def find_in_list(lst: List[str], pred: Callable[[str], bool]) -> Optional[int]:
            for i in range(len(lst)):
                if i not in seen_indexes and pred(lst[i]):
                    seen_indexes.add(i)
                    return i
            return None

        positions_in_input = []

        for value, title in wizard_tokens_with_title:
            idx_in_input = find_in_list(input_tokens_lower,
                                        lambda token: token == value if len(value) == 1 else token.startswith(value))
            if idx_in_input is None:
                return None
            positions_in_input.append((value, title, idx_in_input))

        positions_in_input.sort(key=lambda item: item[2])
        output_fio = Fio('', None, '')

        for i in range(len(positions_in_input)):
            value, title, idx_in_input = positions_in_input[i]
            begin = 0 if i == 0 else idx_in_input
            end = len(input_tokens) if i == len(positions_in_input) - 1 else positions_in_input[i + 1][2]
            expanded_name_part = ' '.join(input_tokens[begin:end])
            output_fio.__setattr__(title, expanded_name_part)

        if not output_fio.first_name or not output_fio.last_name:
            return None

        return output_fio
