from typing import Any, ClassVar, Dict, Optional

import ujson
from aiohttp import BasicAuth

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.base import AbstractInteractionClient


class SenderClient(AbstractInteractionClient):
    """Клиент для запросов в рассылятор."""
    SERVICE = 'sender'
    BASE_URL = settings.SENDER_CLIENT_BASE_URL.strip('/')
    account_slug: ClassVar[str] = settings.SENDER_CLIENT_ACCOUNT_SLUG
    basic_auth_login: ClassVar[str] = settings.SENDER_CLIENT_BASIC_AUTH_LOGIN
    basic_auth_password: ClassVar[str] = settings.SENDER_CLIENT_BASIC_AUTH_PASSWORD

    def _get_session_kwargs(self) -> Dict[str, Any]:
        """Аргументы для конструктора объекта HTTP-сессии."""
        auth = BasicAuth(login=self.basic_auth_login, password=self.basic_auth_password)
        kwargs = super()._get_session_kwargs()
        kwargs['auth'] = auth
        return kwargs

    async def send_transactional_letter(self,
                                        mailing_id: str,
                                        to_email: str,
                                        render_context: Optional[dict] = None,
                                        reply_email: Optional[str] = None) -> str:
        """Отправить письмо транзакционной рассылки в рассыляторе на указанный email."""
        url = self.endpoint_url(f'{self.account_slug}/transactional/{mailing_id}/send')
        params = {'to_email': to_email}
        json = {
            "async": True,
            "args": ujson.dumps(render_context or {}),
        }

        headers = dict()
        if reply_email:
            headers['Reply-To'] = reply_email
        if headers:
            json['headers'] = ujson.dumps(headers)

        resp_json = await self.post(
            interaction_method='send_transactional_letter',
            url=url,
            headers={},
            json=json,
            params=params
        )

        return resp_json['result']['message_id']

    async def campaign_detail(self, mailing_id: str) -> Dict[str, Any]:
        url = self.endpoint_url(f'{self.account_slug}/campaign/{mailing_id}/')
        resp_json = await self.get(interaction_method='campaign_detail', url=url)
        return resp_json

    async def render_transactional_letter(self,
                                          campaign_id: int,
                                          letter_id: int,
                                          render_context: Optional[dict] = None) -> str:
        url = self.endpoint_url(f'{self.account_slug}/render/campaign/{campaign_id}/letter/{letter_id}')
        json = {"params": render_context or {}}
        resp_json = await self.post(interaction_method='render_transactional_letter', url=url, json=json)
        return resp_json['result']
