from asyncio import get_event_loop
from typing import List, Optional

import grpc
from mail.payments.protos.spark_suggest_pb2 import HintRequest, HintResponse  # type: ignore
from mail.payments.protos.spark_suggest_pb2_grpc import ExtHintServiceStub  # type: ignore

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.spark_suggest.entities import SparkSuggestItem


class SparkSuggestClient(AbstractInteractionClient):
    SERVICE = 'spark_suggest'
    BASE_URL = settings.SPARK_SUGGEST_URL
    TVM_ID = None
    TVM_SESSION_CLS = None

    @classmethod
    def _make_grpc_request(cls, request: HintRequest) -> HintResponse:
        with grpc.insecure_channel(cls.BASE_URL) as channel:
            stub = ExtHintServiceStub(channel)
            return stub.Autocomplete(request)

    @classmethod
    def _create_hint_request(cls, query: str, regions: Optional[List[int]] = None) -> HintRequest:
        request = HintRequest(query=query, count=settings.SPARK_SUGGEST_COUNT)
        if regions:
            for r in regions:
                request.regions.append(r)
        return request

    @classmethod
    def _process_response(cls, response: HintResponse) -> List[SparkSuggestItem]:
        return [
            SparkSuggestItem(
                spark_id=item.id,
                name=item.name,
                full_name=item.full_name,
                inn=item.inn,
                ogrn=item.ogrn,
                address=item.address,
                leader_name=item.leader_name,
                region_name=item.region_name
            ) for item in response.values
        ]

    async def get_hint(self, query: str, regions: Optional[List[int]] = None) -> List[SparkSuggestItem]:
        loop = get_event_loop()
        request: HintRequest = self._create_hint_request(query=query, regions=regions)
        response: HintResponse = await loop.run_in_executor(None, self._make_grpc_request, request)
        return self._process_response(response)
