import cachetools
from aiohttp import ClientResponse, ContentTypeError
from aiohttp.client import BasicAuth
from multidict import MultiDict

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.exceptions import TinkoffAddressError, TinkoffDataError, TinkoffError
from mail.payments.payments.utils.helpers import without_none


class TinkoffClient(AbstractInteractionClient):
    SERVICE = 'tinkoff'
    BASE_URL = settings.TINKOFF_URL

    _token_cache: cachetools.TTLCache = cachetools.TTLCache(1, settings.TINKOFF_TOKEN_TTL)

    @staticmethod
    def _to_params(errors: dict) -> MultiDict:
        params: MultiDict = MultiDict({})
        for error in errors:
            params.add('validation_error', error)
        return params

    async def _handle_response_error(self, response):
        try:
            data = await response.json()
        except ContentTypeError:
            content = await response.text()
            with self.logger:
                self.logger.context_push(content=content)
                self.logger.error('Tinkoff decode failed.')
            raise TinkoffError(method=response.method, message='Tinkoff response decode fail.')

        message = data.get('message')
        errors = data.get('errors', {})

        with self.logger:
            self.logger.context_push(
                status=response.status,
                message=message,
                error=data.get('error'),
                error_description=data.get('error_description'),
                errors=errors
            )
            self.logger.error('Tinkoff method failed.')

        if 'адрес не найден' in str(message).lower():
            raise TinkoffAddressError(method=response.method, message=message)

        raise TinkoffDataError(method=response.method, message=message, params=self._to_params(errors))

    async def _process_response(self,
                                response: ClientResponse,
                                interaction_method: str) -> ClientResponse:
        if response.status >= 400 or 'status' in await response.json():
            await self._handle_response_error(response)
        return await response.json()

    async def get_token(self):
        if 'token' not in self._token_cache:
            url = self.endpoint_url('oauth/token')
            response = await self.post(
                'get_token',
                url,
                auth=BasicAuth(
                    login=settings.TINKOFF_BASIC_LOGIN,
                    password=settings.TINKOFF_BASIC_PASSWORD,
                ),
                data={
                    'grant_type': 'password',
                    'username': settings.TINKOFF_USERNAME,
                    'password': settings.TINKOFF_PASSWORD,
                }
            )
            self._token_cache['token'] = response['access_token']
        return self._token_cache['token']

    async def create_merchant(self, merchant: Merchant) -> str:
        assert (
            merchant.organization.ogrn is not None
            and merchant.organization.inn is not None
            and merchant.addresses is not None
            and merchant.ceo is not None
            and merchant.contact is not None
        )
        url = self.endpoint_url('register')
        token = await self.get_token()
        response = await self.post(
            'create_merchant',
            url,
            headers={
                'Authorization': f'Bearer {token}',
            },
            json={
                'terminalTypes': [
                    settings.TINKOFF_TERMINAL_3DS,
                    settings.TINKOFF_TERMINAL_NON_3DS,
                ],
                'mcc': settings.TINKOFF_DEFAULT_MCC,
                'billingDescriptor': merchant.organization.english_name,
                'fullName': merchant.organization.full_name,
                'name': merchant.organization.name,
                'inn': merchant.organization.inn,
                'kpp': merchant.organization.kpp,
                'ogrn': int(merchant.organization.ogrn),
                'addresses': [
                    {
                        'type': address.type,
                        'zip': address.zip,
                        'country': address.country,
                        'city': address.city,
                        'street': address.street + (', ' + address.home if address.home else ''),
                    }
                    for address in merchant.addresses
                ],
                'email': merchant.contact.email,
                'ceo': without_none({
                    'firstName': merchant.ceo.name,
                    'lastName': merchant.ceo.surname,
                    'middleName': merchant.ceo.patronymic,
                    'birthDate': merchant.ceo.birth_date.isoformat() if merchant.ceo.birth_date else None,
                    'phone': merchant.ceo.phone,
                }),
                'siteUrl': merchant.organization.site_url or settings.TINKOFF_DEFAULT_SITE_URL,
            },
        )
        return str(response['shopCode'])
