from decimal import Decimal
from logging import ERROR, INFO
from typing import Any, Dict, Optional, Union

from aiohttp import ClientResponse

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.merchant_oauth import MerchantOAuth
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.subscription import Subscription
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.utils.datetime import utcnow

from .exceptions import TrustInvalidState, TrustOrderAlreadyExists, TrustUidNotFound, UnknownTrustException


class BaseTrustClient(AbstractInteractionClient):
    @classmethod
    def _make_callback_url(cls, *, merchant_uid: Optional[int] = None, order_id: Optional[int] = None) -> str:
        return f"{settings.CALLBACK_ADDRESS}/callback/payment/{merchant_uid}/{order_id}"

    @staticmethod
    def _make_headers(yandexuid: Optional[str], customer_uid: Optional[int]) -> dict:
        headers = {}
        if yandexuid is not None:
            headers['X-YANDEXUID'] = str(yandexuid)
        if customer_uid is not None:
            headers['X-UID'] = str(customer_uid)
        return headers

    @staticmethod
    def make_order_id(uid: int, order_id: int, product_id: int, customer_uid: Optional[int], version: int) -> str:
        return {
            1: f'{settings.TRUST_ORDER_ID_PREFIX}{uid}.{order_id}.{product_id}.{customer_uid or "_"}',
            2: f'{settings.TRUST_ORDER_ID_PREFIX}{uid}-{order_id}-{product_id}-{customer_uid or "_"}'
        }[version]

    @staticmethod
    def _make_product_id(uid: int, inn: str, partner_id: Optional[str], nds: str, service_fee: Optional[int]) -> str:
        parts = [uid, inn, partner_id, nds]
        if service_fee is not None:
            parts.append(service_fee)
        return f'{settings.TRUST_PRODUCT_ID_PREFIX}{".".join(map(str, parts))}'

    @staticmethod
    def _make_pass_params(acquirer: AcquirerType,
                          submerchant_id: Optional[str] = None,
                          oauth: Optional[MerchantOAuth] = None) -> dict:
        result: dict = {}
        if acquirer == AcquirerType.TINKOFF:
            assert submerchant_id
            result['submerchantIdRbs'] = submerchant_id
        elif acquirer == AcquirerType.KASSA:
            assert oauth
            result['oplata_yakassa_data'] = {"merchant_oauth": oauth.decrypted_access_token}
        return result

    @staticmethod
    def _make_afs_params(
        yandexuid: Optional[str] = None,
        payments_service_slug: Optional[str] = None,
        login_id: Optional[str] = None,
        merchant_initiated_transaction: Optional[bool] = False,
    ) -> dict:
        result: dict = {}
        if yandexuid:
            result['yandexuid'] = yandexuid
        if payments_service_slug:
            result['paymentsServiceSlug'] = payments_service_slug
        if login_id:
            result['login_id'] = login_id
        if merchant_initiated_transaction:
            result['request'] = 'mit'
        return result

    @staticmethod
    def _make_refund_order(uid: int,
                           original_order_id: int,
                           product_id: int,
                           amount: Decimal,
                           customer_uid: Optional[int],
                           version: int,
                           ) -> dict:
        trust_order_id = BaseTrustClient.make_order_id(
            uid=uid,
            order_id=original_order_id,
            product_id=product_id,
            customer_uid=customer_uid,
            version=version,
        )
        return {
            'order_id': trust_order_id,
            'delta_qty': str(amount),
        }

    @classmethod
    def _make_product_data(cls, uid: int, partner_id: Optional[str], nds: str, inn: str,
                           service_fee: Optional[int]) -> dict:
        result: Dict[str, Union[str, int]] = {
            'name': f'NDS_{nds}',
            'fiscal_inn': inn,
            'fiscal_nds': nds,
            'fiscal_title': f'NDS_{nds}',
            'product_id': cls._make_product_id(uid, inn, partner_id, nds, service_fee),
            'partner_id': f'{partner_id}',
        }

        if service_fee is not None:
            result['service_fee'] = service_fee

        return result

    @classmethod
    def _make_product_subscription_data(cls, merchant: Merchant, subscription: Subscription) -> dict:
        assert merchant.client_id
        start_ts = int(utcnow().timestamp())

        data = {
            "product_id": subscription.product_id,
            "name": subscription.title,
            "product_type": "subs",
            "subs_period": subscription.period,
            "prices": [
                {
                    "currency": price.currency,
                    "price": f'{price.price}',
                    "region_id": price.region_id,
                    "start_ts": str(start_ts)
                } for price in subscription.prices
            ],
            "fiscal_nds": subscription.nds.value,
            "fiscal_title": subscription.fiscal_title,
            "partner_id": merchant.client_id,
            'aggregated_charging': int(settings.TRUST_AGGREGATED_CHARGING),
            'subs_retry_charging_limit': settings.TRUST_SUBS_RETRY_CHARGING_LIMIT,
            'subs_retry_charging_delay': settings.TRUST_SUBS_RETRY_CHARGING_DELAY,
        }

        if subscription.trial_period:
            data['single_purchase'] = 1
            data['subs_trial_period'] = subscription.trial_period

        return data

    @classmethod
    def _make_subscription_data(cls, order: Order, item: Item) -> dict:
        assert (order.customer_subscription
                and order.customer_subscription.subscription
                and order.order_id
                and item.product_id)

        return {
            "product_id": order.customer_subscription.subscription.product_id,
            "order_id": cls.make_order_id(
                order.uid, order.order_id, item.product_id, order.customer_uid, order.data.version
            ),
            "region_id": order.customer_subscription.region_id,
        }

    async def _handle_response_error(self, response):
        log_level = INFO
        error_data = await response.json()
        status_code = error_data.get('status_code')
        status_desc = error_data.get('status_desc')

        if status_code == 'order_already_exists':
            exc = TrustOrderAlreadyExists(method=response.method, message=status_code)
        elif status_code == 'invalid_state':
            exc = TrustInvalidState(method=response.method, message=status_code)
        elif status_code == 'uid_not_found':
            exc = TrustUidNotFound(method=response.method, message=status_code)
        else:
            log_level = ERROR
            exc = UnknownTrustException(method=response.method, message=status_code)

        with self.logger:
            self.logger.context_push(error_data=error_data, status_code=status_code, status_desc=status_desc)
            self.logger.log(log_level, exc.name)

        raise exc

    def _get_service_token(self, uid: int, acquirer: AcquirerType) -> str:
        acquirer_token = settings.TRUST_SERVICE_TOKEN[acquirer.value]
        return self.get_merchant_setting(uid, 'trust_service_token', acquirer_token)  # type: ignore

    async def _make_request(self,  # type: ignore
                            interaction_method: str,
                            method: str,
                            url: str,
                            *,
                            uid: int,
                            acquirer: AcquirerType,
                            **kwargs: Any,
                            ) -> ClientResponse:
        headers = kwargs.pop('headers', {})
        # User service token has higher priority than default acquirer token.
        headers['X-Service-Token'] = self._get_service_token(uid, acquirer)
        return await super()._make_request(interaction_method, method, url, headers=headers, **kwargs)
