from typing import Dict, Generator, List, Optional

import ujson

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.utils.helpers import split_list

from .base import BaseTrustClient

DictGen = Generator[dict, None, None]


class TrustOrderClient(BaseTrustClient):
    _orders_batch_create_limit: int = settings.TRUST_ORDER_BATCH_CREATE_LIMIT

    async def orders_create(self,
                            uid: int,
                            acquirer: AcquirerType,
                            merchant: Merchant,
                            order: Order,
                            items: List[Item],
                            customer_uid: Optional[int] = None,
                            paymethod_id: Optional[str] = None,
                            service_fee: Optional[int] = None,
                            commission: Optional[int] = None,
                            ) -> List[Dict]:
        """Создать в трасте заказы."""
        trust_orders = list(self._trust_orders_data(merchant=merchant, order=order, items=items, acquirer=acquirer,
                                                    customer_uid=customer_uid, paymethod_id=paymethod_id,
                                                    service_fee=service_fee, commission=commission))
        await self._orders_batch_create(
            uid=uid,
            acquirer=acquirer,
            trust_orders=trust_orders,
            headers=self._make_headers(None, customer_uid),
        )
        return trust_orders

    def _trust_orders_data(self, merchant: Merchant, order: Order, items: List[Item], acquirer: AcquirerType,
                           customer_uid: Optional[int] = None, paymethod_id: Optional[str] = None,
                           service_fee: Optional[int] = None, commission: Optional[int] = None) -> DictGen:
        """Формирует объекты для создания заказов в Трасте."""
        for item in items:
            if item.nds is None:
                raise RuntimeError('Item without NDS')

            assert order.order_id and item.product_id
            assert merchant.organization.inn is not None

            trust_order_id = self.make_order_id(
                merchant.uid,
                order.order_id,
                item.product_id,
                customer_uid,
                order.data.version,
            )

            trust_product_id = self._make_product_id(
                uid=merchant.uid,
                inn=merchant.organization.inn,
                partner_id=merchant.client_id,
                nds=item.nds.value,
                service_fee=service_fee,
            )

            legal = merchant.get_address_by_type('legal')

            # Поле developer_payload используется в двух местах: здесь (на каждой позиции корзины) и в trust/payment.py
            # Сейчас фронт траста читает это поле из первой позиции корзины, поэтому мы задаем это поле здесь,
            # на каждой позиции корзины, а не на самой корзине.
            developer_payload = {
                'merchant': {
                    'schedule_text': merchant.organization.schedule_text,
                    'ogrn': merchant.organization.ogrn,
                    'name': merchant.name,
                    'acquirer': acquirer.value,
                    'legal_address': {
                        'city': legal.city,
                        'country': legal.country,
                        'home': legal.home,
                        'street': legal.street,
                        'zip': legal.zip,
                    } if legal else None
                },
                'selected_card_id': paymethod_id,
                # 'payment_completion_action': 'redirect_always',
            }

            trust_order = {
                'order_id': trust_order_id,
                'product_id': trust_product_id,
                'developer_payload': ujson.dumps(developer_payload),
            }
            if commission is not None:
                trust_order['commission_category'] = str(commission)

            yield trust_order

    async def _orders_batch_create(self,
                                   uid: int,
                                   acquirer: AcquirerType,
                                   trust_orders: List[Dict],
                                   *,
                                   headers: Optional[Dict] = None,
                                   _batch_size: Optional[int] = None,
                                   ) -> List[Dict]:
        """
        Создать список Trust-заказов пачками.
        https://wiki.yandex-team.ru/TRUST/Payments/API/Orders/#sozdaniepachkizakazovdopolnitelnyjjmetod
        Семантика точки - обновлять существующие заказы - если вдруг заказ существует, то ошибки не будет.
        """
        url = self.endpoint_url('orders_batch')
        for batch in split_list(trust_orders, size=_batch_size or self._orders_batch_create_limit):
            data = {'orders': batch}
            await self.post('_orders_batch_create', url, json=data, headers=headers, uid=uid, acquirer=acquirer)
        return trust_orders

    async def _order_create(self, uid: int, acquirer: AcquirerType, order_data: dict,
                            headers: Optional[dict] = None) -> dict:
        """
        Создание Заказа
        https://wiki.yandex-team.ru/TRUST/Payments/API/Orders/#sozdaniezakaza
        """
        url = self.endpoint_url('orders')
        return await self.post(
            '_order_create',
            url,
            json=order_data,
            headers=headers,
            uid=uid,
            acquirer=acquirer,
        )

    async def _order_get(self, uid: int, acquirer: AcquirerType, order_id: int) -> dict:
        """
        Статус Заказа
        https://wiki.yandex-team.ru/TRUST/Payments/API/Orders/#statuszakaza
        """
        url = self.endpoint_url(f'orders/{order_id}')
        return await self.get('_order_get', url, uid=uid, acquirer=acquirer)
