from typing import Dict, Generator, List, Optional, TypedDict, Union, cast

import ujson

from sendr_utils import enum_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import AcquirerType, ReceiptType
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant_oauth import MerchantOAuth
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.utils.helpers import without_none

from .base import BaseTrustClient
from .entities import PaymentMode
from .exceptions import TrustException

DictGen = Generator[dict, None, None]


class PaymentCreateResult(TypedDict):
    purchase_token: str
    trust_payment_id: str


class TrustPaymentClient(BaseTrustClient):
    async def _payment_create(self, uid: int, acquirer: AcquirerType, payment_data: dict,
                              headers: Optional[dict] = None) -> dict:
        """
        Создание Корзины
        https://wiki.yandex-team.ru/TRUST/Payments/API/Baskets/#sozdaniekorziny
        """
        url = self.endpoint_url('payments')
        return await self.post(
            '_payment_create',
            url,
            json=payment_data,
            headers=headers,
            uid=uid,
            acquirer=acquirer,
            params={'show_trust_payment_id': 1},
        )

    async def _payment_get(self, uid: int, acquirer: AcquirerType, purchase_token: str,
                           with_terminal_info: bool = False) -> dict:
        """
        Статус Корзины
        https://wiki.yandex-team.ru/TRUST/Payments/API/Baskets/#statuskorziny
        """
        url = self.endpoint_url(f'payments/{purchase_token}')
        params = {}
        if with_terminal_info:
            params['with_terminal_info'] = 1
        return await self.get('payment_get', url, uid=uid, acquirer=acquirer, params=params)

    async def _payment_start(self, uid: int, acquirer: AcquirerType, purchase_token: str) -> dict:
        """
        Запуск Корзины на оплату
        https://wiki.yandex-team.ru/TRUST/Payments/API/Baskets/#zapuskkorzinynaoplatu
        """
        url = self.endpoint_url(f'payments/{purchase_token}/start')
        return await self.post('_payment_start', url, uid=uid, acquirer=acquirer)

    async def payment_unhold(self, uid: int, acquirer: AcquirerType, purchase_token: str) -> dict:
        """
        Отмена платежа
        https://wiki.yandex-team.ru/TRUST/Payments/API/TwoPhase/#otmenaplatezha
        """
        url = self.endpoint_url(f'payments/{purchase_token}/unhold')
        return await self.post('payment_unhold', url, uid=uid, acquirer=acquirer)

    async def payment_clear(self, uid: int, acquirer: AcquirerType, purchase_token: str) -> dict:
        """
        Клиринг платежа
        https://wiki.yandex-team.ru/TRUST/Payments/API/TwoPhase/#kliringplatezha
        """
        url = self.endpoint_url(f'payments/{purchase_token}/clear')
        return await self.post('payment_clear', url, uid=uid, acquirer=acquirer)

    async def payment_resize(self,
                             uid: int,
                             order: Order,
                             acquirer: AcquirerType,
                             purchase_token: str,
                             item: Item,
                             customer_uid: Optional[int] = None
                             ) -> dict:
        """
        Частичный клиринг платежа
        https://wiki.yandex-team.ru/TRUST/Payments/API/TwoPhase/#chastichnyjjkliring
        """
        assert order.order_id is not None and item.product_id is not None
        trust_order_id = BaseTrustClient.make_order_id(
            uid, order.order_id, item.product_id, customer_uid, order.data.version
        )
        url = self.endpoint_url(f'payments/{purchase_token}/orders/{trust_order_id}/resize')
        # В терминологии Траста qty -- количество товара, amount -- общая сумма.
        trust_data = {
            'qty': str(item.amount),
            'amount': str(item.total_price),
        }
        return await self.post('payment_partional_clear', url, uid=order.uid, json=trust_data, acquirer=acquirer)

    async def _payment_markup(self, uid: int, acquirer: AcquirerType, purchase_token: str, markup_data: dict) -> dict:
        """
        Разметка по способам оплаты
        https://wiki.yandex-team.ru/TRUST/Payments/API/Baskets/#razmetkaposposobamoplaty
        """
        url = self.endpoint_url(f'payments/{purchase_token}/markup')
        return await self.post('_payment_markup', url, uid=uid, acquirer=acquirer, json=markup_data)

    async def payment_create(self,
                             uid: int,
                             acquirer: AcquirerType,
                             order: Order,
                             items: List[Item],
                             template_tag: str,
                             user_email: str,
                             payment_mode: Optional[PaymentMode] = PaymentMode.WEB_PAYMENT,
                             paymethod_id: Optional[str] = 'trust_web_page',
                             return_url: Optional[str] = None,
                             yandexuid: Optional[str] = None,
                             login_id: Optional[str] = None,
                             payments_service_slug: Optional[str] = None,
                             customer_uid: Optional[int] = None,
                             trust_form_name: Optional[str] = None,
                             submerchant_id: Optional[str] = None,
                             selected_card_id: Optional[str] = None,
                             oauth: Optional[MerchantOAuth] = None,
                             payment_completion_action: Optional[Union[str, Dict[str, str]]] = None,
                             ) -> PaymentCreateResult:
        payment_data = self._trust_payment_data(
            uid, acquirer, order, items, return_url, template_tag, user_email, customer_uid, trust_form_name,
            submerchant_id, oauth, payment_mode, paymethod_id, selected_card_id, payment_completion_action,
            payments_service_slug=payments_service_slug,
            yandexuid=yandexuid, login_id=login_id,
        )
        headers = self._make_headers(yandexuid, customer_uid)
        body = await self._payment_create(uid=uid, acquirer=acquirer, payment_data=payment_data, headers=headers)
        if body['status'] == 'success' and body['status_code'] == 'payment_created':
            return {
                'purchase_token': body['purchase_token'],
                'trust_payment_id': body['trust_payment_id'],
            }
        else:
            raise TrustException(method='POST', message='payment_create')

    async def payment_markup(self,
                             uid: int,
                             acquirer: AcquirerType,
                             items: List[Item],
                             purchase_token: str,
                             order: Order,
                             customer_uid: Optional[int],
                             ) -> None:
        data = self._trust_markup_data(items, uid, order, customer_uid)
        assert data, 'Markup data should not be empty'
        body = await self._payment_markup(uid=uid, acquirer=acquirer, purchase_token=purchase_token, markup_data=data)
        if body['status'] != 'success':
            raise TrustException(method='POST', message='payment_markup')

    async def payment_start(self, uid: int, acquirer: AcquirerType, purchase_token: str, without_url: bool) -> Dict:
        body = await self._payment_start(uid=uid, acquirer=acquirer, purchase_token=purchase_token)
        if body['status'] == 'success' and body['payment_status'] == 'started':
            assert without_url or 'payment_url' in body
            return {
                'purchase_token': purchase_token,
                'payment_url': body.get('payment_url')
            }
        else:
            raise TrustException(method='POST', message='payment_start')

    async def payment_status(self, uid: int, acquirer: AcquirerType, purchase_token: str) -> str:
        body = await self.payment_get(uid=uid, acquirer=acquirer, purchase_token=purchase_token)
        return body['payment_status']

    async def payment_get(self, uid: int, acquirer: AcquirerType, purchase_token: str,
                          with_terminal_info: bool = False) -> Dict:
        body = await self._payment_get(uid=uid, acquirer=acquirer, purchase_token=purchase_token,
                                       with_terminal_info=with_terminal_info)
        if body['status'] == 'success':
            return body
        else:
            raise TrustException(method='POST', message='payment_get')

    async def payment_deliver(self, uid: int, acquirer: AcquirerType, purchase_token: str, order: Order,
                              items: List[Item]) -> Dict:
        """
        Чеки на доставку
        https://wiki.yandex-team.ru/TRUST/Payments/API/fiscal_receipts/#chekinadostavku
        """
        assert order.order_id

        orders = list()
        for item in items:
            assert item.product is not None
            orders.append(
                {
                    'fiscal_nds': item.product.nds.value,
                    'order_id': self.make_order_id(
                        uid, order.order_id, cast(int, item.product_id), order.customer_uid, order.data.version
                    )
                }
            )

        url = self.endpoint_url(f'payments/{purchase_token}/deliver')
        return await self.post('payment_deliver', url, uid=uid, json={'orders': orders}, acquirer=acquirer)

    @staticmethod
    def _trust_payment_orders_data(uid: int, order: Order, items: List[Item],
                                   customer_uid: Optional[int] = None) -> DictGen:
        """Формирует объекты для создания корзины(=платежа) в Трасте. Заказы должны быть созданы."""
        for item in items:
            if item.nds is None:
                raise RuntimeError('Expected item\'s NDS is not None')
            assert order.order_id and item.product_id

            trust_order_id = BaseTrustClient.make_order_id(
                uid, order.order_id, item.product_id, customer_uid, order.data.version
            )
            trust_payment_order = {
                'order_id': trust_order_id,
                'fiscal_nds': item.nds.value,
                'fiscal_title': item.name,
                'price': str(item.price),
                'qty': str(item.amount),
            }

            yield trust_payment_order

    def _trust_payment_data(self, uid: int, acquirer: AcquirerType, order: Order, items: List[Item],
                            return_url: Optional[str], template_tag: str, user_email: str,
                            customer_uid: Optional[int] = None, trust_form_name: Optional[str] = None,
                            submerchant_id: Optional[str] = None, oauth: Optional[MerchantOAuth] = None,
                            payment_mode: Optional[PaymentMode] = PaymentMode.WEB_PAYMENT,
                            paymethod_id: Optional[str] = 'trust_web_page',
                            selected_card_id: Optional[str] = None,
                            payment_completion_action: Optional[Union[str, Dict[str, str]]] = None,
                            payments_service_slug: Optional[str] = None,
                            yandexuid: Optional[str] = None,
                            login_id: Optional[str] = None) -> Dict:

        orders = list(self._trust_payment_orders_data(uid=uid, order=order, items=items,
                                                      customer_uid=customer_uid))
        currency = None
        if items:
            if items[-1].currency is None:
                raise RuntimeError('Item without currency')
            currency = items[-1].currency

        developer_payload: Dict[str, Union[str, Dict[str, str]]] = {}
        if trust_form_name:
            developer_payload['form_name'] = trust_form_name
        if selected_card_id:
            developer_payload['selected_card_id'] = selected_card_id
        if payment_completion_action:
            developer_payload['payment_completion_action'] = payment_completion_action

        pass_params = self._make_pass_params(acquirer, submerchant_id, oauth)
        afs_params = self._make_afs_params(
            yandexuid=yandexuid,
            payments_service_slug=payments_service_slug,
            login_id=login_id,
            merchant_initiated_transaction=order.data.without_3ds,
        )
        return without_none({
            'back_url': self._make_callback_url(merchant_uid=uid, order_id=order.order_id),
            'currency': currency,
            'lang': 'ru',
            'orders': orders,
            'pass_params': pass_params,
            'afs_params': afs_params,
            'payment_mode': enum_value(payment_mode),
            'payment_timeout': settings.TRUST_PAYMENT_TIMEOUT,
            'paymethod_id': paymethod_id,
            'return_path': return_url,
            'template_tag': template_tag,
            'user_email': user_email,
            'developer_payload': ujson.dumps(developer_payload),
            'fiscal_expects_delivery': order.data.receipt_type == ReceiptType.PREPAID,
        })

    @staticmethod
    def _trust_markup_data(items: List[Item], uid: int, order: Order, customer_uid: Optional[int]) -> Dict:
        markup_data = {}
        for item in items:
            if item.markup is not None:
                assert order.order_id and item.product_id
                trust_order_id = BaseTrustClient.make_order_id(
                    uid, order.order_id, item.product_id, customer_uid, order.data.version
                )
                markup_data[trust_order_id] = item.markup
        return {'paymethod_markup': markup_data}
