from typing import Optional

from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.interactions.trust.base import BaseTrustClient
from mail.payments.payments.interactions.trust.exceptions import TrustException


class TrustPaymentMethodsClient(BaseTrustClient):
    DEFAULT_TRUST_CURRENCY = 'RUB'

    async def _get_payment_methods(self, uid: int, acquirer: AcquirerType, headers: Optional[dict] = None) -> dict:
        """
        Список доступных способов оплаты
        https://wiki.yandex-team.ru/TRUST/Payments/API/PaymentMethods/#json-obektenabledpaymentmethod
        """
        url = self.endpoint_url('payment-methods')
        params = {'show-plates': 0,
                  'show-enabled': 1,
                  'show-bound': 0,
                  'show_trust_payment_id': 0}
        return await self.get('payment_methods_get', url, headers=headers, params=params, uid=uid, acquirer=acquirer)

    async def get_enabled_currency(self,
                                   uid: int,
                                   acquirer: AcquirerType,
                                   default_currency: str = DEFAULT_TRUST_CURRENCY) -> str:
        headers = self._make_headers(yandexuid=None, customer_uid=uid)
        body = await self._get_payment_methods(uid=uid, acquirer=acquirer, headers=headers)

        if body['status'] == 'success':
            enabled_payment_methods = body.get('enabled_payment_methods')
            if not enabled_payment_methods or len(enabled_payment_methods) > 1:
                self.logger.info('Not found or found more then 1 enabled payments methods in trust')
                return default_currency
            else:
                return enabled_payment_methods[0]['currency']
        else:
            raise TrustException(method='GET', message='payment_methods')
