from typing import Any, Dict, Iterable, Optional

from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant_oauth import MerchantOAuth

from .base import BaseTrustClient
from .exceptions import TrustException


class TrustRefundClient(BaseTrustClient):
    def _make_create_request_data_skeleton(self,
                                           acquirer: AcquirerType,
                                           caption: str,
                                           purchase_token: str,
                                           submerchant_id: Optional[str] = None,
                                           oauth: Optional[MerchantOAuth] = None,
                                           ) -> Dict[str, Any]:
        data: Dict[str, Any] = {
            'orders': [],
            'pass_params': self._make_pass_params(acquirer, submerchant_id, oauth),
            'purchase_token': purchase_token,
            'reason_desc': caption,
        }
        return data

    async def _request_refund_create(self,
                                     uid: int,
                                     acquirer: AcquirerType,
                                     customer_uid: int,
                                     data: Dict[str, Any]
                                     ) -> str:
        result = await self.post(
            'refund_create',
            self.endpoint_url('refunds'),
            json=data,
            headers=self._make_headers(
                yandexuid=None,
                customer_uid=customer_uid,
            ),
            uid=uid,
            acquirer=acquirer,
        )
        if result['status'] == 'success':
            return result['trust_refund_id']
        else:
            raise TrustException(method='POST', message='refund_create')

    async def refund_create(self,
                            uid: int,
                            acquirer: AcquirerType,
                            original_order_id: int,
                            customer_uid: int,
                            caption: str,
                            purchase_token: str,
                            items: Iterable[Item],
                            submerchant_id: Optional[str],
                            oauth: Optional[MerchantOAuth],
                            version: int,
                            ) -> str:
        """
        Создание Возврата
        https://wiki.yandex-team.ru/TRUST/Payments/API/Refunds/#sozdanierefanda
        """

        data = self._make_create_request_data_skeleton(acquirer, caption, purchase_token, submerchant_id, oauth)

        orders = []
        for item in items:
            if item.product_id is None:
                raise RuntimeError('Expected item with product_id')
            orders.append(
                self._make_refund_order(
                    uid=uid,
                    original_order_id=original_order_id,
                    customer_uid=customer_uid,
                    product_id=item.product_id,
                    amount=item.amount,
                    version=version,
                )
            )
        data['orders'] = orders

        return await self._request_refund_create(uid, acquirer, customer_uid, data)

    async def refund_create_single(self,
                                   uid: int,
                                   acquirer: AcquirerType,
                                   customer_uid: int,
                                   caption: str,
                                   purchase_token: str,
                                   quantity: int,
                                   trust_order_id: str,
                                   submerchant_id: Optional[str] = None,
                                   oauth: Optional[MerchantOAuth] = None,
                                   ) -> str:
        """
        Создание Возврата по имеющимся trust_order_id и purchase_token
        https://wiki.yandex-team.ru/TRUST/Payments/API/Refunds/#sozdanierefanda
        """

        data = self._make_create_request_data_skeleton(acquirer, caption, purchase_token, submerchant_id, oauth)

        orders = [{
            'order_id': trust_order_id,
            'delta_qty': str(quantity),
        }]
        data['orders'] = orders

        return await self._request_refund_create(uid, acquirer, customer_uid, data)

    async def _refund_start(self, uid: int, acquirer: AcquirerType, refund_id: str) -> dict:
        """
        Запуск Возврата
        https://wiki.yandex-team.ru/TRUST/Payments/API/Refunds/#zapuskrefanda
        """
        url = self.endpoint_url(f'refunds/{refund_id}/start')
        return await self.post('_refund_start', url, uid=uid, acquirer=acquirer)

    async def refund_get(self, uid: int, acquirer: AcquirerType, refund_id: str) -> dict:
        """
        Статус Рефанда
        https://wiki.yandex-team.ru/TRUST/Payments/API/Refunds/#statusrefanda
        """
        url = self.endpoint_url(f'refunds/{refund_id}')
        return await self.get('refund_get', url, uid=uid, acquirer=acquirer)

    async def refund_start(self, uid: int, acquirer: AcquirerType, refund_id: str) -> str:
        result = await self._refund_start(uid=uid, acquirer=acquirer, refund_id=refund_id)
        if result['status'] in ['success', 'wait_for_notification']:
            return refund_id
        else:
            raise TrustException(method='POST', message='refund_start')
