from datetime import datetime
from typing import Optional

from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.utils.datetime import utcnow

from .base import BaseTrustClient


class TrustSubscriptionClient(BaseTrustClient):
    async def _subscription_create(self,
                                   uid: int,
                                   acquirer: AcquirerType,
                                   subscription_data: dict,
                                   headers: Optional[dict] = None,
                                   ) -> dict:
        """
        Создание Подписки
        https://wiki.yandex-team.ru/TRUST/Payments/API/Subscriptions/#scenarijjsozdanijaioplatypodpiski
        """
        url = self.endpoint_url('subscriptions')
        return await self.post(
            '_subscription_create',
            url,
            json=subscription_data,
            headers=headers,
            uid=uid,
            acquirer=acquirer,
        )

    async def _subscription_get(self, uid: int, acquirer: AcquirerType, order_id: str, headers: dict) -> dict:
        """
        Получение Подписки
        https://wiki.yandex-team.ru/TRUST/Payments/API/Subscriptions/#scenarijjsozdanijaioplatypodpiski
        """
        url = self.endpoint_url(f'subscriptions/{order_id}')
        return await self.get('_subscription_get', url, headers=headers, uid=uid, acquirer=acquirer)

    async def _subscription_cancel(self,
                                   uid: int,
                                   acquirer: AcquirerType,
                                   order_id: str,
                                   finish_ts: datetime,
                                   headers: dict,
                                   ) -> dict:
        """
        Остановка Подписки
        https://wiki.yandex-team.ru/TRUST/Payments/API/Subscriptions/#ostanovkapodpiski
        """
        url = self.endpoint_url(f'subscriptions/{order_id}')
        json = {'finish_ts': finish_ts.timestamp()}
        return await self.put('_subscription_cancel', url, json=json, headers=headers, uid=uid, acquirer=acquirer)

    async def subscription_create(self,
                                  uid: int,
                                  acquirer: AcquirerType,
                                  order: Order,
                                  item: Item,
                                  customer_uid: Optional[int] = None) -> dict:
        subscription_data = self._make_subscription_data(order, item)
        headers = self._make_headers(None, customer_uid)
        return await self._subscription_create(
            uid=uid, acquirer=acquirer, subscription_data=subscription_data, headers=headers)

    async def subscription_get(self,
                               uid: int,
                               acquirer: AcquirerType,
                               order: Order,
                               item: Item,
                               customer_uid: Optional[int] = None,
                               ) -> dict:
        subscription_data = self._make_subscription_data(order, item)
        headers = self._make_headers(None, customer_uid)
        return await self._subscription_get(
            uid=uid, acquirer=acquirer, order_id=subscription_data['order_id'], headers=headers)

    async def subscription_cancel(self,
                                  uid: int,
                                  acquirer: AcquirerType,
                                  order: Order,
                                  item: Item,
                                  finish_ts: Optional[datetime] = None,
                                  customer_uid: Optional[int] = None) -> dict:
        subscription_data = self._make_subscription_data(order, item)
        headers = self._make_headers(None, customer_uid)
        if finish_ts is None:
            finish_ts = utcnow()
        return await self._subscription_cancel(
            uid=uid, acquirer=acquirer, order_id=subscription_data['order_id'], finish_ts=finish_ts, headers=headers)
