from json.decoder import JSONDecodeError
from typing import List, Optional
from uuid import UUID

from aiohttp import ClientResponse, ContentTypeError

from sendr_interactions import exceptions as interaction_errors
from sendr_utils import without_none

from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import YandexPayPartnerType, YandexPayPaymentGatewayType
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.interactions.yandex_pay_admin.entities import Contact, Document
from mail.payments.payments.interactions.yandex_pay_admin.exceptions import get_exception_by_error_code
from mail.payments.payments.interactions.yandex_pay_admin.schemas import UpdateModerationRequestSchema


class YandexPayAdminClient(AbstractInteractionClient):
    SERVICE = 'yandex_pay_admin'
    BASE_URL = settings.YANDEX_PAY_ADMIN_URL.rstrip('/')
    TVM_ID = settings.TVM_YANDEX_PAY_ADMIN_CLIENT_ID

    async def _handle_response_error(self, response: ClientResponse) -> None:
        try:
            error = await response.json()
        except (ContentTypeError, JSONDecodeError):
            error = 'Invalid response body'
            raise interaction_errors.InteractionResponseError(
                status_code=response.status,
                response_status=None,
                params={'error': error},
                service=self.SERVICE,
                method=response.method,
            )

        error_data = error.get('data', {})
        error_code = error_data.get('message')
        error_cls = get_exception_by_error_code(error_code)
        error_params = error_data.get('params')
        raise error_cls(
            status_code=response.status,
            response_status=error_code,
            service=self.SERVICE,
            params=error_params,
            method=response.method,
        )

    async def put_partner(
        self,
        partner_id: UUID,
        name: str,
        uid: int,
        partner_type: YandexPayPartnerType,
        contact: Contact,
        psp_external_id: Optional[str] = None,
        payment_gateway_type: Optional[YandexPayPaymentGatewayType] = None,
        merchant_gateway_id: Optional[str] = None,
        merchant_desired_gateway: Optional[str] = None,
    ) -> None:
        data = without_none({
            'partner_id': str(partner_id),
            'name': name,
            'psp_external_id': psp_external_id,
            'type': partner_type.value,
            'payment_gateway_type': payment_gateway_type.value if payment_gateway_type else None,
            'merchant_gateway_id': merchant_gateway_id,
            'merchant_desired_gateway': merchant_desired_gateway,
            'uid': uid,
            'contact': without_none({
                'email': contact.email,
                'phone': contact.phone,
                'name': contact.name,
                'surname': contact.surname,
                'patronymic': contact.patronymic if contact.patronymic is not None else None,
            }),
        })
        return await self.put(
            interaction_method='put_partner',
            url=self.endpoint_url('/api/v1/partner'),
            json=data,
        )

    async def update_moderation(
        self,
        partner_id: UUID,
        verified: Optional[bool],
        documents: List[Document],
    ) -> None:
        assert UUID(str(partner_id)) == partner_id, 'partner_id contains invalid characters'
        return await self.post(
            interaction_method='update_moderation',
            url=self.endpoint_url(f'/api/v1/partner/{partner_id}/moderation'),
            json=UpdateModerationRequestSchema().dump({
                'verified': verified,
                'documents': documents,
            }).data,
        )
