from typing import Dict, Type

from sendr_interactions import exceptions as interaction_errors


class BaseYaPayAdminError(interaction_errors.InteractionResponseError):
    MESSAGE: str

    def __init__(self, /, message=None, **kwargs):
        super().__init__(message=message or self.MESSAGE, **kwargs)


class PSPExternalIDChangedYaPayAdminError(BaseYaPayAdminError):
    MESSAGE = 'psp_external_id has changed'


class PartnerTypeChangedYaPayAdminError(BaseYaPayAdminError):
    MESSAGE = 'partner_type has changed'


class SchemaValidationYaPayAdminError(BaseYaPayAdminError):
    MESSAGE = 'Schema validation failed'


class PSPExternalIDEmptyYaPayAdminError(BaseYaPayAdminError):
    MESSAGE = 'Payment gateway external id is empty'


class UnknownYaPayAdminError(BaseYaPayAdminError):
    MESSAGE = 'Unknown Yandex Pay Admin error'


_ERROR_CODE_TO_EXCEPTION: Dict[str, Type[BaseYaPayAdminError]] = {
    'PSP_EXTERNAL_ID_CHANGED': PSPExternalIDChangedYaPayAdminError,
    'PSP_EXTERNAL_ID_EMPTY': PSPExternalIDEmptyYaPayAdminError,
    'PARTNER_TYPE_CHANGED': PartnerTypeChangedYaPayAdminError,
    'SCHEMA_VALIDATION_ERROR': SchemaValidationYaPayAdminError,
}


def get_exception_by_error_code(status: str) -> Type[BaseYaPayAdminError]:
    return _ERROR_CODE_TO_EXCEPTION.get(status, UnknownYaPayAdminError)
