import io
from typing import Any, ClassVar

from aiohttp import ClientResponse, ClientTimeout

from mail.payments.payments.conf import settings
from mail.payments.payments.interactions.zora import AbstractZoraClient
from mail.payments.payments.interactions.zora_images.exceptions import ImageTooLargeError


class ZoraImagesClient(AbstractZoraClient):
    SERVICE = 'zora-images'

    TOTAL_TIMEOUT: ClassVar[int] = settings.MERCHANT_IMAGE_HTTP_TOTAL_TIMEOUT
    CONNECTION_TIMEOUT: ClassVar[int] = settings.MERCHANT_IMAGE_HTTP_CONNECTION_TIMEOUT
    IMAGE_MAX_SIZE: ClassVar[int] = settings.MERCHANT_IMAGE_MAX_SIZE

    def __init__(self, *args: Any, **kwargs: Any) -> None:
        super().__init__(*args, **kwargs)
        self._timeout = ClientTimeout(
            total=self.TOTAL_TIMEOUT,
            sock_connect=self.CONNECTION_TIMEOUT,
        )

    async def _process_response(self,
                                response: ClientResponse,
                                interaction_method: str) -> ClientResponse:
        if response.status >= 400:
            await self._handle_response_error(response)
        return response

    async def get_image(self, url: str) -> bytes:
        method = 'get_image'
        response: ClientResponse = await self.get(method, url, response_log_body=False)
        buf = io.BytesIO()
        async for chunk in response.content.iter_any():
            buf.write(chunk)
            if buf.tell() > self.IMAGE_MAX_SIZE:
                raise ImageTooLargeError(method=method)

        return buf.getvalue()
