from typing import Any

from marshmallow import fields
from marshmallow_enum import EnumField

from sendr_utils.schemas.base import BaseSchema
from sendr_utils.schemas.dataclass import BaseDataClassSchema
from sendr_utils.schemas.one_of_schema import OneOfSchema

from mail.payments.payments.core.entities.enums import (
    FunctionalityType, YandexPayPartnerType, YandexPayPaymentGatewayType
)
from mail.payments.payments.core.entities.functionality import (
    PaymentsFunctionalityData, YandexPayFunctionalityData, YandexPayMerchantFunctionalityData,
    YandexPayPaymentGatewayFunctionalityData
)


class FunctionalityPaymentsSchema(BaseDataClassSchema[PaymentsFunctionalityData]):
    type = EnumField(FunctionalityType, by_value=True)


class FunctionalityYandexPayBaseSchema(BaseSchema):
    type = EnumField(FunctionalityType, by_value=True)
    partner_type = EnumField(YandexPayPartnerType, by_value=True)
    partner_id = fields.UUID(required=True)


class FunctionalityYandexPayPaymentGatewaySchema(
    BaseDataClassSchema[YandexPayPaymentGatewayFunctionalityData], FunctionalityYandexPayBaseSchema
):
    gateway_id = fields.String(required=True)
    payment_gateway_type = EnumField(YandexPayPaymentGatewayType, by_value=True, required=True)


class FunctionalityYandexPayMerchantSchema(
    BaseDataClassSchema[YandexPayMerchantFunctionalityData], FunctionalityYandexPayBaseSchema
):
    merchant_gateway_id = fields.String(allow_none=True)
    merchant_desired_gateway = fields.String(allow_none=True)


class FunctionalityYandexPaySchema(OneOfSchema):
    type_field = 'partner_type'
    type_field_remove = True
    type_schemas = {
        YandexPayPartnerType.MERCHANT.value: FunctionalityYandexPayMerchantSchema,
        YandexPayPartnerType.PAYMENT_GATEWAY.value: FunctionalityYandexPayPaymentGatewaySchema,
    }

    def get_obj_type(self, obj: Any) -> str:
        if isinstance(obj, YandexPayPaymentGatewayFunctionalityData):
            return YandexPayPartnerType.PAYMENT_GATEWAY.value
        elif isinstance(obj, YandexPayMerchantFunctionalityData):
            return YandexPayPartnerType.MERCHANT.value
        else:
            raise Exception("Unknown object type: {}".format(obj.__class__.__name__))


class FunctionalitySchema(OneOfSchema):
    type_field = 'type'
    type_field_remove = True
    type_schemas = {
        FunctionalityType.PAYMENTS.value: FunctionalityPaymentsSchema,
        FunctionalityType.YANDEX_PAY.value: FunctionalityYandexPaySchema,
    }

    def get_obj_type(self, obj: Any) -> str:
        if isinstance(obj, YandexPayFunctionalityData):
            return FunctionalityType.YANDEX_PAY.value
        elif isinstance(obj, PaymentsFunctionalityData):
            return FunctionalityType.PAYMENTS.value
        else:
            raise Exception("Unknown object type: {}".format(obj.__class__.__name__))
